using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Identity;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using Microsoft.AspNetCore.Authorization;
using TrackerDeFavorisApi.Models;
using TrackerDeFavorisApi.Services;

namespace TrackerDeFavorisApi.Controllers
{
    [Authorize]
    [Route("api/[controller]")]
    [ApiController]
    public class UserController : ControllerBase
    {
        /*private static List<User> users = new List<User>{
            new User("ValentinP", "azertyu"),
            new User("Khemsyt", "maudeupace")
        };*/

        private readonly BaseContext _context;
        private PasswordHasher<User> _hasher;
        private readonly JwtService _jwts;

        public UserController(BaseContext ctx, PasswordHasher<User> ph, JwtService jwts) {
            _jwts = jwts;
            _context = ctx;
            _hasher = ph;
        }

        // GET: api/<UserController>
        [HttpGet]
        [Authorize(Roles="Admin")]
        public async Task<List<UserWoPassword>> GetUsers()
        {
            List<User> users = await _context.User.ToListAsync();
            List<UserWoPassword>? result = users.Select(u => new UserWoPassword(u)).ToList();
            return result;
        }

        // GET: api/<UserController>
        [HttpGet("admin")]
        [Authorize(Roles="Admin")]
        public IEnumerable<string> GetAdminUsers()
        {
            return _context.User.Where(u => (u.Role == Role.Admin)).Select(u => u.ToString()).ToArray<string>();
        }

        // GET api/<UserController>/5
        [HttpGet("{id}")]
        [Authorize(Roles="Admin")]
        public async Task<ActionResult<UserWoPassword>> GetUser(int id)
        {
            var user = await _context.User.FindAsync(id);

            if (user == null) {
                return NotFound();
            }
            return Ok(new UserWoPassword(user));
        }

        // GET api/<UserController>/find/Name
        [HttpGet("find/{name}")]
        [Authorize]
        public async Task<ActionResult<UserWoPassword>> GetUserByName(string name)
        {
            List<User> user = await _context.User.Where(u => u.Pseudo.Equals(name)).ToListAsync();

            if (user == null) {
                return NotFound();
            }

            return Ok(new UserWoPassword(user[0]));
           
           
           // return _context.User.Where(u => (u.Pseudo.Contains(name))).Select(u => u.ToString()).ToArray<string>();
        }

        // GET api/<UserController>/search/Name
        [HttpGet("search/{name}")]
        [Authorize]
        public async Task<ActionResult<UserWoPassword>> GetUsersByName(string name)
        {
            List<User> user = await _context.User.Where(u => u.Pseudo.Contains(name)).ToListAsync();

            if (user == null) {
                return NotFound();
            }

            return Ok(new UserWoPassword(user[0]));
           
           
           // return _context.User.Where(u => (u.Pseudo.Contains(name))).Select(u => u.ToString()).ToArray<string>();
        }

        // POST api/<UserController>/register
        [HttpPost("register")]
        [AllowAnonymous]
        public async Task<ActionResult<User>> PostRegister(UserInfo ui)
        {
            int a = _context.User.Where(u =>(u.Pseudo == ui.Pseudo)).Count();
            Console.WriteLine("a:"+a);

            if (a==0){
                User user = new User(ui.Pseudo, "");

                // Hachage du mot de passe
                user.Password = _hasher.HashPassword(user, ui.Password);
            
                _context.User.Add(user);
                await _context.SaveChangesAsync();

                return CreatedAtAction(nameof(GetUser), new { id = user.Id }, user);
            }
            else {
                return StatusCode(409,"Le nom d'utilisateur existe déjà. Veuillez en choisir un autre.");
            }

            // faire la querie (regarder si l'utilisateur existe avec un .count) si zero faire la suite si pas zero return StatusCode(409,"Le nom d'utilisateur existe déjà. Veuillez en choisir un autre.");
            
        }

        // POST api/<UserController>/login
        [HttpPost("login")]
        [AllowAnonymous]
        public async Task<ActionResult<UserAndJwt>> PostLogin(UserInfo ui)
        {
            User? user = await _context.User.FirstOrDefaultAsync(u => u.Pseudo.CompareTo(ui.Pseudo) == 0);

            if (user == null) {
                return NotFound();
            }

            if (user.Pseudo.CompareTo(ui.Pseudo) != 0) {
                return Unauthorized("Pseudo ou Mot de passe incorrect");
            }

            if (_hasher.VerifyHashedPassword(user, user.Password, ui.Password) != PasswordVerificationResult.Success) {
                return Unauthorized("Pseudo ou Mot de passe incorrect");
            }

            return Ok(new UserAndJwt(user, JwtService.toto(user.Id, user.Pseudo, user.Role,_jwts.JwtKey)));
        }

        // PUT api/<UserController>/5
        [HttpPut("{id}")]
        [Authorize(Roles="Admin")]
        public async Task<ActionResult<User>> Put(int id, UserUpdate userUpdate)
        {
            User? user = await _context.User.FindAsync(id);

            if (user == null) {
                return NotFound();
            }

            if (userUpdate.Pseudo != "") {
                user.Pseudo = userUpdate.Pseudo;
            }

            if (userUpdate.Password != "") {
                user.Password = _hasher.HashPassword(user, userUpdate.Password);
            }

            user.Role = userUpdate.Role;

            _context.Entry(user).State = EntityState.Modified;

            try {
                await _context.SaveChangesAsync();
            } catch (DbUpdateConcurrencyException) {
                return StatusCode(500, "Erreur de concurrence");
            }

            return Ok(user);
        }

        // DELETE api/<UserController>/5
        [HttpDelete("{id}")]
        [Authorize(Roles="Admin")]
        public async Task<IActionResult> DeleteUser(int id)
        {
            // on récupère le user que l'on souhaite supprimer
            User? user = await _context.User.FindAsync(id);
            if (user == null)
            {
                return NotFound();
            }
            // on indique a notre contexte que l'objet a été supprimé
            _context.User.Remove(user);
            // on enregistre les modifications
            await _context.SaveChangesAsync();
            // on retourne un code 204 pour indiquer que la suppression a bien eu lieu
            return NoContent();
        }
    }
}
