using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using Microsoft.IdentityModel.Tokens;
using Microsoft.AspNetCore.Authorization;
using TrackerDeFavorisApi.Models;

namespace TrackerDeFavorisApi.Controllers
{
    [Authorize]
    [Route("api/[controller]")]
    [ApiController]
    public class FilmController : ControllerBase
    {
        private readonly BaseContext _context;

        public FilmController(BaseContext ctx) {
            _context = ctx;
        }

        // GET: api/<FilmController>
        [HttpGet]
        [AllowAnonymous]
        public async Task<List<Film>> GetFilms()
        {
            return await _context.Film.ToListAsync();
        }

        // GET api/<FilmController>/search
        [HttpGet("search/{Title}")]
        [AllowAnonymous]
        public ActionResult<IEnumerable<Film>> GetFilmFromTitle(string Title)
        {
            IEnumerable<Film> listFilms = _context.Film.Where(f => f.Title!.ToLower().Contains(Title.ToLower()));

            if (listFilms.IsNullOrEmpty())
            {
                return NotFound();
            }

            return Ok(listFilms);
        }

        [HttpGet("{id}")]
        [AllowAnonymous]
        public async Task<ActionResult<Film>> GetFilmFromId(int id)
        {
            var film = await _context.Film.FindAsync(id);

            if (film == null) {
                return NotFound();
            }

            return Ok(film);
        }

        // GET api/<FilmController>/info
        [HttpGet("info")]
        [AllowAnonymous]
        public ActionResult<IEnumerable<Film>> GetFilmInfos([FromQuery] int[] ids)
        {
            IEnumerable<Film> listFilms = _context.Film.Where(u => (ids.Contains(u.Id)));

            if (listFilms.IsNullOrEmpty())
            {
                return NotFound();
            }

            return Ok(listFilms);
        }

        // DELETE api/<FilmController>/5
        [HttpDelete("{id}")]
        [Authorize(Roles="Admin")]
        public async Task<IActionResult> DeleteFilm(int id)
        {
            // on récupère le user que l'on souhaite supprimer
            Film? film = await _context.Film.FindAsync(id);
            if (film == null)
            {
                return NotFound();
            }
            // on indique a notre contexte que l'objet a été supprimé
            _context.Film.Remove(film);
            // on enregistre les modifications
            await _context.SaveChangesAsync();
            // on retourne un code 204 pour indiquer que la suppression a bien eu lieu
            return NoContent();
        }
    }
}