using System.Linq.Expressions;
using System.Reflection;
using System.Security.Claims;
using FavTracker.Models;
using Microsoft.AspNetCore.Components.Authorization;
using Microsoft.AspNetCore.Components.Server.ProtectedBrowserStorage;

namespace FavTracker.Services;

public class AuthProvider : AuthenticationStateProvider {
    private readonly ProtectedLocalStorage _sessionStorage;
    
    public AuthProvider(ProtectedLocalStorage protectedLocalStorage) {
        _sessionStorage = protectedLocalStorage;        
    }
    

    public override async Task<AuthenticationState> GetAuthenticationStateAsync() {
        var user = await _sessionStorage.GetAsync<User>("User");
        var token = await _sessionStorage.GetAsync<string>("Token");

        if (user.Value != null && token.Value != null) {
            var claim = GenerateClaimsPrincipal(user.Value, token.Value);
            return new AuthenticationState(claim);
        }

        return new AuthenticationState(new ClaimsPrincipal(new ClaimsIdentity()));
    }

    public ClaimsPrincipal GenerateClaimsPrincipal(User user, string token) {
        var claims = new [] {
            new Claim("Id", user.Id.ToString()),
            new Claim(ClaimTypes.Name, user.Pseudo),
            new Claim(ClaimTypes.Role, user.Role.ToString()),
            new Claim("Token", token)
        };

        ClaimsIdentity identity = new ClaimsIdentity(claims, "custom");
        ClaimsPrincipal claimsPrincipal = new ClaimsPrincipal(identity);

        return claimsPrincipal;
    }

    public async Task Login(User user, string token) {
        await _sessionStorage.SetAsync("User", user);
        await _sessionStorage.SetAsync("Token", token);

        ClaimsPrincipal claim = GenerateClaimsPrincipal(user, token);
        NotifyAuthenticationStateChanged(Task.FromResult(new AuthenticationState(claim)));
    }

    public async Task Logout() {
        await _sessionStorage.DeleteAsync("User");
        await _sessionStorage.DeleteAsync("Token");

        ClaimsPrincipal claimDisconnected = new ClaimsPrincipal(new ClaimsIdentity());
        NotifyAuthenticationStateChanged(Task.FromResult(new AuthenticationState(claimDisconnected)));
    }

    public async Task<bool> IsConnected()
    {
        var authState = await GetAuthenticationStateAsync();
        var user = authState.User;
        return user.Identity?.IsAuthenticated ?? false;
    }

    public async Task<UserAndJwt?> GetUserAndJwt()
    {
        var authState = await GetAuthenticationStateAsync();
        var user = authState.User;

        if (user != null)
        {
            var userAndJwt = new UserAndJwt(
                new User(user.FindFirst(ClaimTypes.Name)?.Value ?? "", ""),
                user.FindFirst("Token")?.Value ?? ""
            );

            if (int.TryParse(user.FindFirst("Id")?.Value, out int userId))
            {
                if (userAndJwt != null && userAndJwt.User != null) {
                    userAndJwt.User.Id = userId;
                }
            }

            return userAndJwt;
        }

        return null;
    }
}