#include <SDL2/SDL_render.h>
#include <SDL2/SDL_stdinc.h>
#include <SDL2/SDL_timer.h>
#include <SDL2/SDL_ttf.h>
#include <SDL2/SDL_video.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <time.h>

#include "board.h"
#include "display.h"
#include "display_effects.h"
#include "entity.h"
#include "input.h"
#include "move.h"
#include "menu.h"
#include "move.h"
#include "mcts.h"
#include "textures.h"

#define FIRE_FRAMES 101
#define MLG_FRAMES 151

void reset_selection(move_t move, piece_t * piece1, piece_t * piece2, board_t moves) {
    reset_move(move);

    get_blank_board(moves);

    *piece1 = get_blank_piece();
    *piece2 = get_blank_piece();
}

void select_with_coord(bool positioning, board_t board, board_t moves, move_t move, player_t * curr_player) {
    if (positioning) {

        if ((move[0].x == -1 && move[0].y != -1) && (curr_player->stash[move[0].y])) {
            highlight_possible_placements(board, moves, *curr_player);
        } else if ((move[0].x != -1 && move[0].y != -1) &&
                (get_piece_from_coord(board, move[0]).owner_id == curr_player->id)) {
            highlight_possible_placements(board, moves, *curr_player);
        } else {
            reset_move(move);
        }

    } else {
        if ((move[0].x != -1 && move[0].y != -1) && (get_piece_from_coord(board, move[0]).owner_id == curr_player->id)) {
            // Si la source est valide, on affiche les coups possibles
            get_highlight_board(board, moves, move[0], curr_player);
        } else {
            reset_move(move);
        }

    }
}

int main(int argc, char ** argv) {
    bool is_player0_filled = 1;
    bool is_player1_filled = 1;

    (void) argc;
    (void) argv;

    srand(time(0));

    SDL_Window * window = NULL;
    SDL_Renderer * renderer = NULL;

    SDL_DisplayMode screen;

    // Initialisation de la SDL & Gestion des échecs

    if (SDL_Init(SDL_INIT_VIDEO) != 0) {
        exit_sdl(0, "Error when initializing SDL", window, renderer);
    }

    // Récupération des paramètres de l'écran

    SDL_GetCurrentDisplayMode(0, &screen);

    // Création de la fenêtre

    window = SDL_CreateWindow(
        "Stratego",             // Nom de la fenêtre
        SDL_WINDOWPOS_CENTERED,     // Position x
        SDL_WINDOWPOS_CENTERED,     // Position y
        screen.w * 0.66,            // Largeur de la fenêtre
        screen.h * 0.66,            // Hauteur de la fenêtre
        SDL_WINDOW_RESIZABLE
    );

    if (window == NULL) {
        exit_sdl(0, "Error when creating the window", window, renderer);
    }

    // Création du renderer

    renderer = SDL_CreateRenderer(window, -1, SDL_RENDERER_ACCELERATED | SDL_RENDERER_PRESENTVSYNC);

    if (renderer == NULL) {
        exit_sdl(0, "Error when creating the renderer", window, renderer);
    }

    // Création de TTF : Affichage du texte

    if (TTF_Init() < 0) {
        exit_sdl(0, "Couldn't initialize SDL TTF", window, renderer);
    }

    // Chargement des textures

    SDL_Texture * banner, * large_banner, * table, * board_background, *** pieces, ** fire, ** mlg, ** romeo, ** antonin;
    load_all_textures(&banner, &large_banner, &table, &board_background, &pieces, &fire, &mlg, &romeo, &antonin, window, renderer);

    // Affichage du menu

    int nb_opponents = menu(table, banner, large_banner, window, renderer);

    // Création du plateau

    board_t board, moves, new_board;
    get_start_board(board);
    get_blank_board(moves);

    // Création des joueurs

    player_t player0 = get_player(player_0);
    player_t player1 = get_player(player_1);

    player_t * curr_player = &player0;
    player_t * opp_player = &player1;

    bool player0_positioned = 0;
    bool player1_positioned = (nb_opponents == 0);

    if (is_player1_filled || nb_opponents == 0) {
        place_player1_piece_randomly(board, &player1);
    }

    if (is_player0_filled) {
        board_t init_board;
        get_blank_board(init_board);

        place_player1_piece_randomly(init_board, &player0);

        for (int i = 0; i < 4; ++i) {
            for (int j = 0; j < BOARD_SIZE; ++j) {
                board[BOARD_SIZE-1-i][j] = init_board[i][j]; // Juste pour les tests
            }
        }
    }

    // Création des systèmes de coordonnées

    coord_t coord_hovered = {-1, -1}, prev_coord_hovered = {-1, -1};
    int mouse_x = 0, mouse_y = 0;

    move_t move = {{-1, -1}, {-1, -1}};

    bool prev_ok_button_hovered = 0, ok_button_hovered = 0;
    bool ok_button_clicked = 0;

    piece_t piece1, piece2;
    int fight_result;

    bool empty_stash = 0;
    bool curr_player_loser = 0;
    bool move_valid = 0;

    bool positioning = 1;

    bool move_displayed = 0;

    int fire_frame = 0, mlg_frame = 0;
    float anim_step = 0;

    bool mlg_on = 0;
    bool mlg_ending = 0;

    bool fight_step1 = 0, fight_step2 = 0, fight_step3 = 0;

    double old_time, new_time;
    double frame = 0;

    // Affichage du jeu

    bool program_on = (nb_opponents != -1), update = 1;

    SDL_Event event;

    while (program_on && !curr_player_loser) {
        // Gestion des images

        old_time = new_time;
        new_time = SDL_GetTicks();

        frame += (new_time - old_time);

        if (move_displayed && frame > 50) {
            frame = 0;
            update = SDL_TRUE;
        }

        // Gestion des événements

        if (SDL_PollEvent(&event)) {
            switch (event.type) {
                case SDL_QUIT:
                    program_on = SDL_FALSE;
                    break;

                case SDL_MOUSEMOTION:
                    mouse_x = event.button.x;
                    mouse_y = event.button.y;

                    prev_coord_hovered = coord_hovered;
                    coord_hovered = get_tile_coord(curr_player->id, mouse_x, mouse_y, window);

                    prev_ok_button_hovered = ok_button_hovered;
                    ok_button_hovered = is_ok_button(curr_player->id, mouse_x, mouse_y, window);

                    if (!move_displayed || frame > 50) {
                        if (prev_coord_hovered.x != coord_hovered.x || prev_coord_hovered.y != coord_hovered.y) {
                            update = SDL_TRUE;
                        }

                        if (prev_ok_button_hovered != ok_button_hovered) {
                            update = SDL_TRUE;
                        }
                    }
                    break;
                
                case SDL_MOUSEBUTTONUP:
                    switch (event.button.button) {
                        case SDL_BUTTON_LEFT:
                            
                            if (move_displayed) {
                                SDL_SetWindowTitle(window, "Stratego");

                                if (fight_step2) {
                                    fight_step2 = 0;
                                    
                                    if (fight_result) {
                                        fight_step3 = 1;
                                    } else {
                                        move_displayed = 0;
                                    }
                                } else {
                                    move_displayed = 0;
                                }

                                if (!fight_step3) {
                                    reset_selection(move, &piece1, &piece2, moves);
                                }

                            } else {
                                ok_button_clicked = is_ok_button(curr_player->id, event.button.x, event.button.y, window);

                                if (move[0].x == -1 && move[0].y == -1) {
                                    // On récupère la première coordonnée
                                    move[0] = get_tile_coord(curr_player->id, event.button.x, event.button.y, window);

                                    // On affiche tous les coups possibles
                                    select_with_coord(positioning, board, moves, move, curr_player);

                                } else {
                                    // On récupère la deuxième coordonnée

                                    move[1] = get_tile_coord(curr_player->id, event.button.x, event.button.y, window);

                                    // On effectue le déplacement

                                    if (positioning) {

                                        if (move[0].x == -1 && move[1].x != -1 && move[1].y != -1) {
                                            // Placement d'une pièce de la réserve sur le terrain
                                            place_piece_on_phase_drop(moves, board, move, curr_player);
                                        } else if (move[0].x != -1 && move[0].y != -1) {
                                            if (move[1].x == -1) {
                                                // Replacement d'une pièce du terrain dans la réserve
                                                undo_placement(board, move[0], curr_player);
                                            } else {
                                                // Déplacement d'une pièce déjà placée sur le terrain
                                                change_placement_on_drop_phase(move, moves, board, curr_player);
                                            }
                                        }

                                        empty_stash = is_player_stash_empty(*curr_player);
                                        reset_selection(move, &piece1, &piece2, moves);

                                    } else if (move[1].x != -1 && move[1].y != -1) {
                                        // Affichage des deux pièces attaquées

                                        piece1 = get_piece_from_coord(board, move[0]);
                                        piece2 = get_piece_from_coord(board, move[1]);

                                    } else {
                                        reset_selection(move, &piece1, &piece2, moves);
                                    }

                                }

                                update = SDL_TRUE;
                            }
                            break;
                        
                        case SDL_BUTTON_RIGHT:
                            reset_selection(move, &piece1, &piece2, moves);
                            update = SDL_TRUE;
                            break;

                        default:
                            break;
                    }
                    break;

                case SDL_WINDOWEVENT:
                    if (event.window.event == SDL_WINDOWEVENT_RESIZED) {
                        update = SDL_TRUE;
                    }
                    break;
                
                default:
                    break;
            }
        }

        if (ok_button_clicked) {
            if (positioning && empty_stash) {
                // Prise en compte

                if (curr_player->id == player_0) {
                    player0_positioned = 1;
                } else if (curr_player->id == player_1) {
                    player1_positioned = 1;
                }

                // Réinitialisation des compteurs pour chaque joueur

                if (!is_player1_filled) {
                    player0 = get_player(player_0);
                    player1 = get_player(player_1);
                }

                if (nb_opponents != 0) {
                    if (curr_player->id == player0.id) {
                        curr_player = &player1;
                        opp_player = &player0;
                    } else {
                        curr_player = &player0;
                        opp_player = &player1;
                    }
                }

                // Passage à la partie combat

                positioning = (!player0_positioned || !player1_positioned);

                if (!positioning) {
                    player0 = get_player(player_0);
                    player1 = get_player(player_1);
                }

                if (!positioning && nb_opponents == 0) {
                    init_mcts(board, player1, player0);
                }

            }

            empty_stash = 0;
            ok_button_hovered = 0;
            ok_button_clicked = 0;

            update = SDL_TRUE;
        }

        if (!positioning && !move_displayed) {
            if (nb_opponents == 0 && curr_player->id == player_1) {
                copy_board(board, new_board);

                mcts_take_move(new_board, curr_player, opp_player, &move, &fight_result);

                curr_player = &player0;
                opp_player = &player1;

                piece1 = get_piece_from_coord(board, move[0]);
                piece2 = get_piece_from_coord(board, move[1]);

                copy_board(new_board, board);

                move_displayed = 1;
            } else {
                if ((move[0].x != -1 && move[0].y != -1) && (move[1].x != -1 && move[1].y != -1)) {
                    move_valid = take_move(board, move, curr_player, opp_player, 0, &fight_result);

                    if (move_valid && !positioning) {
                        if (curr_player->id == player0.id) {
                            curr_player = &player1;
                            opp_player = &player0;
                        } else {
                            curr_player = &player0;
                            opp_player = &player1;
                        }

                        mlg_on = (piece1.rank == spy && piece2.rank == marshal);

                        if (mlg_on) {
                            mlg_ending = 1;
                        }

                        fire_frame = 0;
                        mlg_frame = 0;

                        move_displayed = 1;
                        fight_step1 = 1;

                        curr_player_loser = is_player_loser(*curr_player, board);
                    } else {
                        if (!move_valid) {
                            reset_selection(move, &piece1, &piece2, moves);
                        }
                    }
                }
            }
        }

        // Gestion de l'affichage

        if (update) {
            if (move_displayed) {
                display(table, board_background, pieces, banner, large_banner, board, moves, curr_player->id, move[0], coord_hovered,
                        &player0, &player1, positioning, ok_button_hovered, empty_stash, 0, window, renderer);

                if (piece2.owner_id != player_nil && piece1.owner_id != piece2.owner_id) {

                    if (fight_step1) {
                        display_piece(pieces[piece1.owner_id][NB_PIECES], board, moves, curr_player->id, move[0], move[0], coord_hovered, window, renderer);
                        display_piece(pieces[piece2.owner_id][NB_PIECES], board, moves, curr_player->id, move[1], move[1], coord_hovered, window, renderer);

                        display_max_pieces(pieces[piece1.owner_id][NB_PIECES], pieces[piece2.owner_id][NB_PIECES],
                                           mouse_x, mouse_y, window, renderer);

                        display_mlg(fire[fire_frame], window, renderer);
                        fire_frame = (fire_frame + 1) % FIRE_FRAMES;                        

                        if (fire_frame == FIRE_FRAMES / 5) {
                            fight_step1 = 0;
                            fight_step2 = 1;
                        }

                        update = SDL_FALSE;
                    } else if (fight_step2) {
                        display_piece(pieces[piece1.owner_id][piece1.rank], board, moves, curr_player->id, move[0], move[0], coord_hovered, window, renderer);
                        display_piece(pieces[piece2.owner_id][piece2.rank], board, moves, curr_player->id, move[1], move[1], coord_hovered, window, renderer);

                        display_max_pieces(pieces[piece1.owner_id][piece1.rank], pieces[piece2.owner_id][piece2.rank],
                                           mouse_x, mouse_y, window, renderer);

                        if (mlg_on) {
                            // Animation spéciale lorsque l'espionne attaque le maréchal
                            SDL_SetWindowTitle(window, "MLG Str@t3g0");
                            display_mlg(mlg[mlg_frame], window, renderer);
                            mlg_frame = (mlg_frame + 1) % MLG_FRAMES;
                        } else {
                            display_mlg(fire[fire_frame], window, renderer);
                            fire_frame = (fire_frame + 1) % FIRE_FRAMES;   
                        }

                        update = SDL_FALSE;         
                    } else if (fight_step3) {
                        display_merge_pieces(pieces[piece1.owner_id][piece1.rank], pieces[piece2.owner_id][piece2.rank], fight_result, anim_step, window, renderer);

                        anim_step += 0.03125;

                        if (anim_step == 1) {
                            fight_step3 = 0;
                            move_displayed = 0;
                            anim_step = 0;

                            reset_selection(move, &piece1, &piece2, moves);

                            update = SDL_TRUE;
                        }
                    }

                } else if (piece1.owner_id != player_nil && piece2.owner_id == player_nil) {
                    display_piece(pieces[piece1.owner_id][NB_PIECES], board, moves, curr_player->id, move[1], move[0], coord_hovered, window, renderer);
                }

                display_banner_side(banner, curr_player->id, window, renderer);

                SDL_RenderPresent(renderer);
                
            } else {
                display(table, board_background, pieces, banner, large_banner, board, moves, curr_player->id, move[0], coord_hovered,
                    &player0, &player1, positioning, ok_button_hovered, empty_stash, 1, window, renderer);
                update = SDL_FALSE;   
            }
        }
    }

    // Ecran de victoire

    if (curr_player_loser) {
        victory(table, romeo, antonin, large_banner, pieces[opp_player->id][0], opp_player, mlg_ending, window, renderer);
    }

    // Fermeture de la SDL

    destroy_all_textures(banner, large_banner, table, board_background, pieces, fire, mlg, romeo, antonin);

    IMG_Quit();
    TTF_Quit();

    exit_sdl(1, "Normal ending", window, renderer);

    // Libération des variables

    return EXIT_SUCCESS;
}