#include <SDL2/SDL_render.h>
#include <SDL2/SDL_ttf.h>
#include <stdbool.h>

#include "display.h"
#include "display_count.h"
#include "display_effects.h"
#include "entity.h"
#include "textures.h"

void display_count_text(TTF_Font * font, int count, SDL_Rect piece_dest, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect txt_dest = {0};          // Rectangle définissant où le texte doit être déposé dans le renderer

    // Importation de la police

    char text_quantity[4];
    
    SDL_Color color = {0, 0, 0, 255}; // Couleur du texte
    
    SDL_Texture * text_texture = NULL;

    // Affichage du texte

    sprintf(text_quantity, "x%d", count);

    text_texture = get_text_texture(text_quantity, font, color, window, renderer);

    SDL_QueryTexture(text_texture, NULL, NULL, &txt_dest.w, &txt_dest.h);

    txt_dest.x = piece_dest.x + 1.1 * piece_dest.w;
    txt_dest.y = piece_dest.y - piece_dest.h / 8;

    SDL_RenderCopy(renderer, text_texture, NULL, &txt_dest);

    SDL_DestroyTexture(text_texture);
}

void display_count_titles(SDL_Texture * banner, TTF_Font * font, SDL_Rect piece_dest, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect
        source = {0},
        banner_dest = {0};

    // Affichage de la bannière

    SDL_QueryTexture(banner, NULL, NULL, &source.w, &source.h);

    banner_dest.w = piece_dest.w;
    banner_dest.h = piece_dest.h / NB_PIECES;

    banner_dest.x = piece_dest.x;
    banner_dest.y = 0.25 * piece_dest.y;

    char text_quantity[18];
    sprintf(text_quantity, "Pieces restantes");

    display_banner(banner, font, text_quantity, banner_dest, window, renderer);
}

void display_count_button_text(TTF_Font * font, SDL_Rect button, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect txt_dest = {0};          // Rectangle définissant où le texte doit être déposé dans le renderer

    // Importation de la police
    
    SDL_Color color = {255, 255, 255, 255}; // Couleur du texte
    
    SDL_Texture * text_texture = NULL;

    // Affichage du texte

    text_texture = get_text_texture("OK", font, color, window, renderer);

    SDL_QueryTexture(text_texture, NULL, NULL, &txt_dest.w, &txt_dest.h);

    txt_dest.x = button.x + (button.w - txt_dest.w) / 2;
    txt_dest.y = button.y + (button.h - txt_dest.h) / 2;

    SDL_RenderCopy(renderer, text_texture, NULL, &txt_dest);

    SDL_DestroyTexture(text_texture);
}

void display_count_piece(TTF_Font * font, SDL_Texture * texture, player_t * player, player_id curr_player,
                         coord_t coord, coord_t coord_selected, coord_t coord_hovered, int rank, int count, bool positioning,
                         SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect
        source = {0},               // Rectangle définissant la zone de la texture à récupérer
        window_dimensions = {0},    // Rectangle définissant la fenêtre
        cell_dest = {0},
        piece_dest = {0};          // Rectangle définissant où la zone source doit être déposée dans le renderer

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h); // Dimensions de la fenêtre

    // On récupère la texture correspondant à la pièce

    SDL_QueryTexture(texture, NULL, NULL, &source.w, &source.h);

    // On fixe les dimensions de la destination

    int length = 0.9 * window_dimensions.h; // Taille du plateau

    int zone_h = length - length / 16;

    // On détermine ensuite la taille d'une cellule

    cell_dest.w = (window_dimensions.w - length) / 2.0 - length / 16.0;
    cell_dest.h = zone_h / NB_PIECES;

    cell_dest.x = length / 32;
    cell_dest.y = window_dimensions.h - length + rank * cell_dest.h;

    if (player->id == player_1) { // Liste du joueur
        cell_dest.x += length + (window_dimensions.w - length) / 2;
    }

    if (positioning && coord.x == coord_selected.x && coord.y == coord_selected.y && player->id == curr_player) {
        highlight_tile(0, 0, 0, 64, cell_dest, renderer);
    }

    if (positioning && coord.x == coord_hovered.x && coord.y == coord_hovered.y && player->id == curr_player) {
        highlight_tile(0, 0, 0, 32, cell_dest, renderer);
    }

    // On ajuste le zoom de la pièce

    float zoom = (float) cell_dest.h / source.h;

    // On la place enfin dans la zone

    piece_dest.w = source.w * zoom;
    piece_dest.h = source.h * zoom;

    piece_dest.x = length / 16;
    piece_dest.y = window_dimensions.h - length + rank * cell_dest.h;

    if (player->id == player_1) { // Liste du joueur
        piece_dest.x += length + (window_dimensions.w - length) / 2;
    }

    SDL_RenderCopy(renderer, texture, &source, &piece_dest);

    display_count_text(font, count, piece_dest, window, renderer);
}

void display_count(SDL_Texture *** textures, SDL_Texture * banner, SDL_Texture * large_banner,
                   player_id curr_player, player_t * player1, player_t * player2,
                   coord_t coord_selected, coord_t coord_hovered,
                   bool positioning, bool ok_button_hovered, bool empty_stash,
                   SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect
        window_dimensions = {0},    // Rectangle définissant la fenêtre
        stash = {0},
        button = {0};          

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h); // Dimensions de la fenêtre

    // Affichage du rectangle

    int length = 0.9 * window_dimensions.h; // Taille du plateau

    SDL_SetRenderDrawColor(renderer, 255, 255, 255, 64);

    stash.w = (window_dimensions.w - length) / 2.0 - length / 16.0;
    stash.h = length - length / 16;

    button.w = stash.w;
    button.h = length / 8;

    // Importation de la police

    TTF_Font * font = NULL;
    font = TTF_OpenFont("../fonts/GreatVibes-Regular.ttf", stash.h / 16);  // La police à charger, la taille désirée
    if (font == NULL) {
        exit_sdl(0, "Can't load font", window, renderer);
    } 

    TTF_SetFontStyle(font, TTF_STYLE_NORMAL);

    // Réserve du joueur du bas

    stash.x = length / 32;
    stash.y = window_dimensions.h - length;

    button.x = stash.x;
    button.y = (window_dimensions.h - button.h) / 2;

    if (!positioning || curr_player == player1->id) {

        display_banner(large_banner, font, NULL, stash, window, renderer);
        display_count_titles(banner, font, stash, window, renderer);

    } else {

        display_banner(banner, font, "OK", button, window, renderer);

        if (empty_stash) {
            if (ok_button_hovered) {
                highlight_tile(213, 30, 30, 128, button, renderer);
            }
        } else {
            highlight_tile(0, 0, 0, 128, button, renderer);
        }

    }

    // Réserve du joueur du haut

    stash.x += stash.w + length + 2 * stash.x;
    button.x = stash.x;

    if (!positioning || curr_player == player2->id) {

        display_banner(large_banner, font, NULL, stash, window, renderer);
        display_count_titles(banner, font, stash, window, renderer);

    } else {

        display_banner(banner, font, "OK", button, window, renderer);

        if (empty_stash) {
            if (ok_button_hovered) {
                highlight_tile(30, 76, 213, 128, button, renderer);
            }
        } else {
            highlight_tile(0, 0, 0, 128, button, renderer);
        }

    }

    // Fermeture de la police

    TTF_CloseFont(font);

    // Réouverture de la police avec un changement de taille

    font = TTF_OpenFont("../fonts/GreatVibes-Regular.ttf", stash.h / NB_PIECES);  // La police à charger, la taille désirée
    if (font == NULL) {
        exit_sdl(0, "Can't load font", window, renderer);
    } 

    TTF_SetFontStyle(font, TTF_STYLE_NORMAL);

    // Affichage des pièces

    coord_t coord;

    for (int i = 0; i < NB_PIECES; ++i) {
        coord.x = -1;
        coord.y = i;

        if (!positioning || curr_player == player1->id) {
            display_count_piece(font, textures[player1->id][i], player1, curr_player,
                                coord, coord_selected, coord_hovered, i, player1->stash[i], positioning, window, renderer);
        }

        if (!positioning || curr_player == player2->id) {
            display_count_piece(font, textures[player2->id][i], player2, curr_player,
                                coord, coord_selected, coord_hovered, i, player2->stash[i], positioning, window, renderer);    
        }        
    }

    TTF_CloseFont(font);
}