#include <SDL2/SDL.h>
#include <SDL2/SDL_image.h>
#include <SDL2/SDL_rect.h>
#include <SDL2/SDL_ttf.h>
#include <SDL2/SDL_log.h>
#include <SDL2/SDL_render.h>
#include <SDL2/SDL_video.h>
#include <stdbool.h>
#include <stdio.h>

#include "display.h"
#include "display_count.h"
#include "board.h"
#include "entity.h"

void highlight_tile(int r, int g, int b, int a, SDL_Rect rectangle, SDL_Renderer * renderer) {
    SDL_SetRenderDrawColor(renderer, r, g, b, a);
    SDL_RenderFillRect(renderer, &rectangle);
}

void display_piece(SDL_Texture * texture, board_t board, board_t moves, player_id curr_player,
                   coord_t coord, coord_t coord_selected, coord_t coord_hovered,
                   SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect
        source = {0},               // Rectangle définissant la zone de la texture à récupérer
        window_dimensions = {0},    // Rectangle définissant la fenêtre
        cell = {0},                 // Rectangle définissan la case
        destination = {0};          // Rectangle définissant où la zone source doit être déposée dans le renderer

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h); // Dimensions de la fenêtre

    // On récupère la texture correspondant à la pièce

    SDL_QueryTexture(texture, NULL, NULL, &source.w, &source.h);

    // On fixe les dimensions de la destination

    int length = 0.9 * window_dimensions.h;         // Taille du plateau
    float tile_size = (float) length / BOARD_SIZE;  // Taille d'une case

    int x_mid = window_dimensions.w / 2;
    int y_mid = window_dimensions.h / 2;

    // Dimensions de la case

    cell.w = tile_size;
    cell.h = tile_size;

    cell.x = x_mid - (length / 2.0) + coord.y * tile_size;
    cell.y = y_mid - (length / 2.0) + coord.x * tile_size;

    if (get_piece_from_coord(moves, coord).owner_id == curr_player) {
        highlight_tile(0, 128, 0, 128, cell, renderer);
    }

    if (coord.x == coord_selected.x && coord.y == coord_selected.y) {
        highlight_tile(255, 255, 255, 128, cell, renderer);
    }

    if (coord.x == coord_hovered.x && coord.y == coord_hovered.y &&
        ((get_piece_from_coord(board, coord).owner_id == curr_player) || (get_piece_from_coord(moves, coord).owner_id == curr_player))) {
        highlight_tile(255, 255, 255, 64, cell, renderer);
    }

    // Dimensions de la pièce

    if (texture != NULL) {
        float zoom = 0.85 * tile_size / source.h; // Zoom de la pièce

        destination.w = source.w * zoom;
        destination.h = source.h * zoom;

        destination.x = x_mid - (length / 2.0) + (tile_size - destination.w) / 2 + coord.y * tile_size;
        destination.y = y_mid - (length / 2.0) + (tile_size - destination.h) / 2 + coord.x * tile_size;

        // On affiche ensuite la texture

        SDL_RenderCopy(renderer, texture, &source, &destination);   
    }

}

void display_board(SDL_Texture * table, SDL_Texture * background, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect
        source = {0},               // Rectangle définissant la zone de la texture à récupérer
        window_dimensions = {0},    // Rectangle définissant la fenêtre
        destination = {0};          // Rectangle définissant où la zone source doit être déposée dans le renderer

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h); // Dimensions de la fenêtre

    // Affichage du fond

    SDL_QueryTexture(table, NULL, NULL, &source.w, &source.h); // Dimensions de l'image

    destination = window_dimensions;

    SDL_RenderCopy(renderer, table, &source, &destination);

    // Affichage du plateau

    int length = 0.9 * window_dimensions.h; // Taille du plateau

    destination.w = length;
    destination.h = length;

    int x_mid = window_dimensions.w / 2; // Point du milieu
    int y_mid = window_dimensions.h / 2;

    destination.x = x_mid - destination.w / 2;
    destination.y = y_mid - destination.h / 2;
    
    SDL_QueryTexture(background, NULL, NULL, &source.w, &source.h);

    SDL_RenderCopy(renderer, background, &source, &destination);
}

void display(SDL_Texture * table, SDL_Texture * background, SDL_Texture *** textures, SDL_Texture * banner, SDL_Texture * large_banner,
             board_t board, board_t moves,
             player_id curr_player, coord_t coord_selected, coord_t coord_hovered, player_t * player1, player_t * player2,
             bool positioning, bool ok_button_hovered, bool empty_stash, bool board_display,
             SDL_Window * window, SDL_Renderer * renderer) {
    SDL_RenderClear(renderer);
    SDL_SetRenderDrawBlendMode(renderer, SDL_BLENDMODE_BLEND);

    SDL_Rect window_dimensions = {0};

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h);
    
    display_board(table, background, window, renderer);

    coord_t coord;

    for (int i = 0; i < BOARD_SIZE; ++i) {
        for (int j = 0; j < BOARD_SIZE; ++j) {
            coord.x = i;
            coord.y = j;

            piece_t piece = get_piece_from_coord(board, coord);

            if (piece.rank == piece_nil) {
                display_piece(NULL, board, moves, curr_player, coord, coord_selected, coord_hovered, window, renderer);
            } else if (!board_display || piece.owner_id != curr_player) {
                display_piece(textures[piece.owner_id][NB_PIECES], board, moves, curr_player, coord, coord_selected, coord_hovered, window, renderer);
            } else {
                display_piece(textures[piece.owner_id][piece.rank], board, moves, curr_player, coord, coord_selected, coord_hovered, window, renderer);
            }
        }
    }

    if (positioning || window_dimensions.w >= 1.5 * window_dimensions.h) {
        display_count(textures, banner, large_banner, curr_player, player1, player2, coord_selected, coord_hovered, positioning, ok_button_hovered, empty_stash, window, renderer);
    }


    if (board_display) {
        SDL_RenderPresent(renderer);
    }
}

void exit_sdl(char ok, char const * msg, SDL_Window * window, SDL_Renderer * renderer) {
    char msg_formatted[255];
    int l;

    if (!ok) {
        strncpy(msg_formatted, msg, 250);
        l = strlen(msg_formatted);
        strcpy(msg_formatted + l, " : %s\n");
        SDL_Log(msg_formatted, SDL_GetError());
    }

    if (renderer != NULL) {
        SDL_DestroyRenderer(renderer);
        renderer = NULL;
    }

    if (window != NULL) {
        SDL_DestroyWindow(window);
        window = NULL;
    }

    SDL_Quit();

    if (!ok) {
        exit(EXIT_FAILURE);
    }
}