#include "board.h"
#include "entity.h"
#include "utils.h"
#include "move.h"
#include "list.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

bool_t is_player_loser(player_t player, board_t board) {
    return ((player.stash[flag] == 0) || (get_number_of_moving_piece_in_stash(player) == SUM_MOVING_PIECE) || (possible_moves(board, &player)->length == 0));
}

bool_t is_coord_valid(coord_t coord) {
    return !((coord.x < 0) || (coord.y < 0) || (coord.x >= BOARD_SIZE) || (coord.y >= BOARD_SIZE));
}

void print_board(board_t board){
    int i,j;
    for (i = 0; i < BOARD_SIZE; i++){
        for (j = 0; j < BOARD_SIZE; j++){
            printf(" %d %d |", board[i][j].owner_id, board[i][j].rank);
        }
        printf("\n");
    }
}

bool_t is_cell_in_bounds(coord_t coord){
    return ((coord.x >=0) && (coord.y >= 0) && (coord.x < BOARD_SIZE) && (coord.y < BOARD_SIZE));
}

piece_t get_piece_from_coord(board_t board, coord_t coord) {
    if (is_cell_in_bounds(coord)){
    return board[coord.x][coord.y];
    }
    else{
        fprintf(stderr,"Cell not in bounds\n");
        return get_blank_piece();
    }
}

void get_blank_board(board_t board) {
    for (int i = 0; i <BOARD_SIZE; i++) {
        for (int j = 0; j<BOARD_SIZE; j++) {
            board[i][j] = get_blank_piece();
        }
    }
}

void get_start_board(board_t board) {
    get_blank_board(board);
    piece_t barrier_piece = get_blank_piece();
    barrier_piece.owner_id = player_barrier;
    for (int i = 4; i<6; i++) {
        board[i][2] = barrier_piece;
        board[i][3] = barrier_piece;
        board[i][6] = barrier_piece;
        board[i][7] = barrier_piece;
    }   
}

void copy_piece(board_t board_dst, piece_rank rank, player_id id, int x, int y){
    board_dst[x][y].rank = rank;
    board_dst[x][y].owner_id =id;
}

list_t * move_all(board_t board, list_t * list, int dx, int dy, int x, int y, player_id id){
    bool_t bool = 1;
    int i = 1;
    move_t move;
    while ((bool) && ((x + i*dx)>=0) && ((y + i*dy)>=0) && ((x + i*dx)<BOARD_SIZE) && ((y + i*dy)<BOARD_SIZE)){
        if ((board[x+i*dx][y+i*dy].owner_id != player_nil)){
            bool = 0;
        }
        if ((board[x+i*dx][y+i*dy].owner_id != id) && (board[x+i*dx][y+i*dy].owner_id != player_barrier)){
            create_move(move,x,y,x+i*dx,y+i*dy);
            list = list_add(list,move);
        }
        i++;
    }
    return list;
}

list_t * move_one(board_t board,list_t * list, int dx, int dy, int x, int y, player_id id){
    move_t move;
    if (((x + dx)>=0) && ((y + dy)>=0) && ((x + dx)<BOARD_SIZE) && ((y + dy)<BOARD_SIZE)){
        if ((board[x+dx][y+dy].owner_id != id) && (board[x+dx][y+dy].owner_id != player_barrier)){
            create_move(move,x,y,x+dx,y+dy);
            list = list_add(list,move);
        }
    }
    return list;
}

list_t * possible_moves_for_piece(coord_t coord, board_t board, list_t * list, player_t * player){
    piece_t piece = get_piece_from_coord(board,coord);
    player_id id = piece.owner_id;
    piece_rank rank = piece.rank;
    if (id == player->id){
        switch (rank){
            case bomb:
            case flag:
                break;
            case scout:
                list = move_all(board,list,-1,0,coord.x,coord.y,id);
                list = move_all(board,list,1,0,coord.x,coord.y,id);
                list = move_all(board,list,0,-1,coord.x,coord.y,id);
                list = move_all(board,list,0,1,coord.x,coord.y,id);
                break;
            default:
                list = move_one(board,list,-1,0,coord.x,coord.y,id);
                list = move_one(board,list,1,0,coord.x,coord.y,id);
                list = move_one(board,list,0,-1,coord.x,coord.y,id);
                list = move_one(board,list,0,1,coord.x,coord.y,id);
                break;
        }
    }
    return list;
}

list_t * possible_moves(board_t board, player_t * player){
    list_t * list = create_list();
    list_t * list_of_next_pieces;
    int i,j;
    coord_t coord;
    for (i=0;i<BOARD_SIZE;i++){
        for (j=0;j<BOARD_SIZE;j++){
            if (board[i][j].owner_id == player->id){
                coord.x = i;
                coord.y = j;
                list_of_next_pieces = create_list();
                list_of_next_pieces  = possible_moves_for_piece(coord,board,list_of_next_pieces,player);
                list = list_merge(list,list_of_next_pieces);
            }
        }
    }
    return list;
}

void get_highlight_board(board_t board, board_t new_board, coord_t coord, player_t * player){
    list_t * list = create_list();
    node_t * node;
    piece_t piece = get_piece_from_coord(board, coord);
    int dst_x, dst_y;
    list = possible_moves_for_piece(coord, board, list, player);
    node = list->head;
    while (node != NULL){
        dst_x = (node->move)[1].x;
        dst_y = (node->move)[1].y;
        copy_piece(new_board,piece.rank,piece.owner_id,dst_x,dst_y);
        node = node->next;
    }
}

void _place_random_in_raw_range(board_t board, player_t * player, piece_rank rank, int a, int b) { //[a, b[
    int x, y;

    piece_t placed_piece = {player->id, rank, 0}; 

    while (player->stash[rank] != 0) {
        x = rand()%(b-a) + a;
        y = rand()%BOARD_SIZE;

        if (board[x][y].owner_id == player_nil) {
            board[x][y] = placed_piece;
            player->stash[rank] --;
        }
    }
}

void place_player1_piece_randomly(board_t board, player_t * player) {
    _place_random_in_raw_range(board, player, flag, 0, 2);
    _place_random_in_raw_range(board, player, bomb, 0, 3);
    _place_random_in_raw_range(board, player, scout, 1, 4);
    _place_random_in_raw_range(board, player, engineer, 1, 3);

    piece_t placed_piece;
    placed_piece.owner_id = player->id;

    for (int i = 0; i<4; i++) {
        for (int j = 0; j<BOARD_SIZE; j++) {

            if (board[i][j].owner_id == player_nil) {
                piece_rank rank;
                do {
                    rank = rand()%NB_PIECES;

                } while (player->stash[rank] <= 0);

                player->stash[rank]--;
                placed_piece.rank = rank;
                placed_piece.has_attacked = 0;
                board[i][j] = placed_piece;
            }

        }
    }
}

void highlight_possible_placements(board_t board, board_t new_board, player_t player) {
    int i,j;
    if (player.id == player_0){
        for (i=BOARD_SIZE-DEPLOYEMENT_ZONE_SIZE;i<BOARD_SIZE;i++){
            for (j=0;j<BOARD_SIZE;j++){
                if (board[i][j].owner_id != player_0){
                    new_board[i][j].owner_id = player_0;
                }
            }
        }
    }
    if (player.id == player_1){
        for (i=0;i<DEPLOYEMENT_ZONE_SIZE;i++){
            for (j=0;j<BOARD_SIZE;j++){
                if (board[i][j].owner_id != player_1){
                    new_board[i][j].owner_id = player_1;
                }
            }
        }
    }
}

void copy_board(board_t src, board_t dst) {
    for (int i = 0; i<BOARD_SIZE; i++) {
        for (int j = 0; j<BOARD_SIZE; j++) {
            dst[i][j].has_attacked = src[i][j].has_attacked;
            dst[i][j].owner_id = src[i][j].owner_id;
            dst[i][j].rank = src[i][j].rank;
        }
    }
}

bool_t is_move_valid(board_t available_move, move_t move) {
    coord_t dst = move[1];
    return (available_move[dst.x][dst.y].owner_id != player_nil);
}

bool_t is_move_attacking(board_t board, move_t move, player_t * player) {
    coord_t dst = move[1];
    piece_t dst_piece = get_piece_from_coord(board, dst);

    return ((dst_piece.owner_id != player->id) && (dst_piece.owner_id != player_nil));
}

void _regular_move(board_t board, coord_t dst, coord_t src, piece_t src_piece) {
    board[dst.x][dst.y] = src_piece;
    board[src.x][src.y] = get_blank_piece();
}

bool_t is_piece_able_to_take_strictly(piece_t src_piece, piece_t dst_piece) {
    // Si le démineur defuse la bombe
    if ((src_piece.rank == engineer) && (dst_piece.rank == bomb)) {
        return 1;
    }

    // Si l'espionne attaque le marechal
    if ((src_piece.rank == spy) && (dst_piece.rank == marshal)) {
        return 1;
    }

    // Si elle gagne "naturellement"
    if (src_piece.rank > dst_piece.rank) {
        return 1;
    }

    return 0;
}

void place_piece_on_phase_drop(board_t available_move, board_t board, move_t move, player_t * player) { // src.x = -1 src.y = rank
    coord_t src = move[0];
    coord_t dst = move[1];

    piece_rank rank = src.y;

    piece_t selected_piece = {player->id, rank, 0};

    if ((player->stash[rank] > 0) && (available_move[dst.x][dst.y].owner_id == player->id)) {
        player->stash[rank] -= 1;
        board[dst.x][dst.y] = selected_piece;
    }
}

void undo_placement(board_t board,coord_t src, player_t * player){
    piece_t piece = get_piece_from_coord(board, src);
    if (piece.owner_id != player->id){
        fprintf(stderr,"Tente de déplacer quelque chose pas à lui\n");
    }
    else{
        board[src.x][src.y] = get_blank_piece();
        player->stash[piece.rank] += 1;
    }
}

void change_placement_on_drop_phase(move_t move, board_t available_move, board_t board, player_t * player){
    move_t move_for_place;
    piece_t piece = get_piece_from_coord(board,move[0]);
    move_for_place[0].x = -1;
    move_for_place[0].y = piece.rank;
    move_for_place[1].x = move[1].x;
    move_for_place[1].y = move[1].y;
    place_piece_on_phase_drop(available_move, board, move_for_place, player);
    undo_placement(board, move[0], player);   
}

// Retourne un bool qui indique si une pièce à bougé et modifie le board en conséquence
// Result = -1 l'attaquant à perdu
// Result = 0 égalité
// Resulat = 1 l'atanquant à gagné
bool_t take_move(board_t board, move_t move, player_t * attacking_player, player_t * defending_player, bool_t bypass_verif, int * result) {
    coord_t src = move[0];
    coord_t dst = move[1];
    board_t available_move;
    get_blank_board(available_move);
    
    if (!bypass_verif) {
        get_highlight_board(board, available_move, src, attacking_player);
    }

    if (bypass_verif || is_move_valid(available_move, move)) {
        
        piece_t src_piece = get_piece_from_coord(board, src);
        piece_t dst_piece = get_piece_from_coord(board, dst);
        
        bool_t is_move_atk = is_move_attacking(board, move, attacking_player);

        if (is_move_atk){
            src_piece.has_attacked = 1;
            dst_piece.has_attacked = 1;
        }
        // Si l'attaque fonctionne
        if (!is_move_atk || is_piece_able_to_take_strictly(src_piece, dst_piece)) {
            _regular_move(board, dst, src, src_piece);
            if (is_move_atk) {
                defending_player->stash[dst_piece.rank] -= 1; 
            }
            set_to_if_not_null(result, 1);
        } else {
            // S'il y a cross kill
            if (src_piece.rank == dst_piece.rank) {
                attacking_player->stash[src_piece.rank] -= 1; 
                defending_player->stash[dst_piece.rank] -= 1; 
                board[src.x][src.y] = get_blank_piece();
                board[dst.x][dst.y] = get_blank_piece();
                set_to_if_not_null(result, 0);
            } else { // Sinon celui qui attaque est plus faible que le defenseur
                attacking_player->stash[src_piece.rank] -= 1;
                board[src.x][src.y] = get_blank_piece();
                set_to_if_not_null(result, -1);
            }
        }
        return 1;
    } else {
        return 0;
    }
}

int cmp_piece(piece_t piece1,piece_t piece2){
    int piece1_value = piece1.rank + piece1.owner_id*100 + piece1.has_attacked*1000;
    int piece2_value = piece2.rank + piece2.owner_id*100 + piece2.has_attacked*1000;
    return piece1_value - piece2_value;
}

int board_cmp(board_t board1, board_t board2) {
    for (int i = 0; i < BOARD_SIZE; i++) {
        for (int j = 0; j < BOARD_SIZE; j++) {
            int cmp_res = cmp_piece(board1[i][j],board2[i][j]);
            if (cmp_res != 0) {
                return cmp_res;
            } 
        }
    }
    return 0;
}