#include "player.h"
#include "piece.h"
#include <stdlib.h>
#include <SDL2/SDL.h>
#include <SDL2/SDL_timer.h>

/**
 * @brief Crée un nouveau timer pour le joueur.
 * 
 * @return timer_player_t Le nouveau timer initialisé.
 */
timer_player_t create_timer() {
    timer_player_t timer;

    timer.start_ticks = 0;
    timer.paused_ticks = 0;

    timer.paused = 0;
    timer.started = 0;

    return timer;
}

/**
 * @brief Démarre le timer.
 * 
 * @param timer Pointeur vers le timer à démarrer.
 */
void start_timer(timer_player_t * timer) {
    timer->started = 1;
    timer->paused = 0;
    timer->start_ticks = SDL_GetTicks();
}

/**
 * @brief Arrête le timer.
 * 
 * @param timer Pointeur vers le timer à arrêter.
 */
void stop_timer(timer_player_t * timer) {
    timer->started = 0;
    timer->paused = 0;
}

/**
 * @brief Met en pause le timer.
 * 
 * @param timer Pointeur vers le timer à mettre en pause.
 */
void pause_timer(timer_player_t * timer) {
    if (timer->started && !timer->paused) {
        timer->paused = 1;
        timer->paused_ticks = SDL_GetTicks() - timer->start_ticks;
    }
}

/**
 * @brief Relance le timer après une pause.
 * 
 * @param timer Pointeur vers le timer à relancer.
 */
void unpause_timer(timer_player_t * timer) {
    if (timer->paused) {
        timer->paused = 0;
        timer->start_ticks = SDL_GetTicks() - timer->paused_ticks;
        timer->paused_ticks = 0;
    }
}

/**
 * @brief Récupère le nombre de ticks écoulés depuis le démarrage du timer.
 * 
 * @param timer Le timer dont on veut connaître le nombre de ticks écoulés.
 * @return int Le nombre de ticks écoulés.
 */
int get_ticks(timer_player_t timer) {
    int ticks;

    if (timer.started) {
        if (timer.paused) {
            ticks = timer.paused_ticks;
        } else {
            ticks = SDL_GetTicks() - timer.start_ticks;
        }
    } else {
        ticks = 0;
    }

    return ticks;
}

/**
 * @brief Crée un nouveau joueur.
 * 
 * @return player_t* Pointeur vers le nouveau joueur créé.
 */
player_t * create_player() {
    player_t * player = malloc(sizeof(player_t));
    for (int i = 0; i < MAX_STASH_PIECE; i++) {
        player->available_drop_piece[i] = get_blank_piece();
    }

    return player;
}

/**
 * @brief Récupère l'index d'une pièce dans le tableau des pièces du joueur.
 * 
 * @param player Le joueur contenant le tableau de pièces.
 * @param piece La pièce dont on veut connaître l'index.
 * @return int L'index de la pièce dans le tableau, ou MAX_STASH_PIECE-1 si non trouvé.
 */
int get_index_of_piece_in_player_drop_piece(player_t * player, piece_t piece) {
    int i = 0;
    while ((i < MAX_STASH_PIECE-2) && (player->available_drop_piece[i].id != piece.id)) {
        i += 1;
    }

    return i;
}

/**
 * @brief Ajoute une pièce au tableau de pièces du joueur.
 * 
 * @param player Le joueur auquel on ajoute la pièce.
 * @param piece La pièce à ajouter.
 */
void add_drop_piece_to_player(player_t * player, piece_t piece) {
    int i = get_index_of_piece_in_player_drop_piece(player, get_blank_piece());
    player->available_drop_piece[i] = swap_piece(piece);
}

/**
 * @brief Retire une pièce du tableau de pièces du joueur.
 * 
 * @param player Le joueur auquel on retire la pièce.
 * @param piece La pièce à retirer.
 */
void remove_drop_piece_to_player(player_t * player, piece_t piece) {
    int i = get_index_of_piece_in_player_drop_piece(player, piece);
    player->available_drop_piece[i] = get_blank_piece();
}

/**
 * @brief Récupère l'ID du joueur opposé.
 * 
 * @param player Le joueur dont on veut connaître l'ID opposé.
 * @return int L'ID du joueur opposé.
 */
player_id get_opposite_player_id(player_t player) {
    return (player.id)?player_0:player_1;
}