#include "piece.h"
#include "player.h"
#include <stdlib.h>

#define MOVES_NB_ROWS 4
#define MOVES_NB_COLS 3

/** 
 * @brief Valeurs des pièces non promues. 
 */
const int piece_values[] = {0, 5000, 10, 8, 6, 4, 3, 2, 1};

/** 
 * @brief Valeurs des pièces promues. 
 */
const int piece_values_promoted[] = {0, 5000, 12, 12, 6, 6, 6, 6, 6};

/** 
 * @brief Déplacements possibles pour chaque type de pièce non promue. 
 */
const move_type piece_moves[PIECE_COUNT][MOVES_NB_ROWS][MOVES_NB_COLS] = {
    { // Blank
        {move_nil, move_nil, move_nil},
        {move_nil, move_nil, move_nil},
        {move_nil, move_nil, move_nil},
        {move_nil, move_nil, move_nil},
    },
    { // King
        {move_nil, move_nil, move_nil},
        {move_one, move_one, move_one},
        {move_one, move_nil, move_one},
        {move_one, move_one, move_one},
    },
    { // Rook
        {move_nil, move_nil, move_nil},
        {move_nil, move_all, move_nil},
        {move_all, move_nil, move_all},
        {move_nil, move_all, move_nil},
    },
    { // Bishop
        {move_nil, move_nil, move_nil},
        {move_all, move_nil, move_all},
        {move_nil, move_nil, move_nil},
        {move_all, move_nil, move_all},
    },
    { // Gold General
        {move_nil, move_nil, move_nil},
        {move_one, move_one, move_one},
        {move_one, move_nil, move_one},
        {move_nil, move_one, move_nil},
    },
    { // Silver General
        {move_nil, move_nil, move_nil},
        {move_one, move_one, move_one},
        {move_nil, move_nil, move_nil},
        {move_one, move_nil, move_one},
    },
    { // Knight
        {move_nil, move_nil, move_nil},
        {move_nil, move_nil, move_nil},
        {move_nil, move_nil, move_nil},
        {move_nil, move_nil, move_nil},
    },
    { // Spear
        {move_nil, move_nil, move_nil},
        {move_nil, move_all, move_nil},
        {move_nil, move_nil, move_nil},
        {move_nil, move_nil, move_nil},
    },
    { // Pawn
        {move_nil, move_nil, move_nil},
        {move_nil, move_one, move_nil},
        {move_nil, move_nil, move_nil},
        {move_nil, move_nil, move_nil},
    },
};

/** 
 * @brief Déplacements possibles pour chaque type de pièce promue. 
 */
const move_type piece_moves_promoted[PIECE_COUNT][MOVES_NB_ROWS][MOVES_NB_COLS] = {
    { // Blank
        {move_nil, move_nil, move_nil},
        {move_nil, move_nil, move_nil},
        {move_nil, move_nil, move_nil},
        {move_nil, move_nil, move_nil},
    },
    { // King
        {move_nil, move_nil, move_nil},
        {move_one, move_one, move_one},
        {move_one, move_nil, move_one},
        {move_one, move_one, move_one},
    },
    { // Rook
        {move_nil, move_nil, move_nil},
        {move_one, move_all, move_one},
        {move_all, move_nil, move_all},
        {move_one, move_all, move_one},
    },
    { // Bishop
        {move_nil, move_nil, move_nil},
        {move_all, move_one, move_all},
        {move_one, move_nil, move_one},
        {move_all, move_one, move_all},
    },
    { // Gold General
        {move_nil, move_nil, move_nil},
        {move_one, move_one, move_one},
        {move_one, move_nil, move_one},
        {move_nil, move_one, move_nil},
    },
    { // Silver General
        {move_nil, move_nil, move_nil},
        {move_one, move_one, move_one},
        {move_one, move_nil, move_one},
        {move_nil, move_one, move_nil},
    },
    { // Knight
        {move_nil, move_nil, move_nil},
        {move_one, move_one, move_one},
        {move_one, move_nil, move_one},
        {move_nil, move_one, move_nil},
    },
    { // Spear
        {move_nil, move_nil, move_nil},
        {move_one, move_one, move_one},
        {move_one, move_nil, move_one},
        {move_nil, move_one, move_nil},
    },
    { // Pawn
        {move_nil, move_nil, move_nil},
        {move_one, move_one, move_one},
        {move_one, move_nil, move_one},
        {move_nil, move_one, move_nil},
    },
};

/**
 * @brief Récupère la valeur d'une pièce.
 * 
 * @param piece La pièce dont on veut connaître la valeur.
 * @return int La valeur de la pièce.
 */
int get_piece_value(piece_t piece) {
    return piece.is_piece_promoted ? piece_values_promoted[piece.id] : piece_values[piece.id];
}

const move_type (*reversed_moves(const move_type (*moves)[MOVES_NB_COLS]))[MOVES_NB_COLS] {
    move_type (*reversed_moves)[MOVES_NB_COLS] = malloc(MOVES_NB_ROWS * sizeof(*reversed_moves));

    for (int i = 0; i < MOVES_NB_ROWS; i++) {
        for (int j = 0; j < MOVES_NB_COLS; j++) {
            reversed_moves[MOVES_NB_ROWS - i - 1][j] = moves[i][j];
        }
    }

    return reversed_moves;
}


/**
 * @brief Récupère les mouvements possibles pour une pièce.
 * 
 * @param piece La pièce dont on veut connaître les mouvements possibles.
 * @return const move_type (*)[MOVES_NB_COLS] Pointeur vers le tableau des mouvements possibles.
 */
const move_type (*get_piece_moves(piece_t piece))[MOVES_NB_COLS] {
    const move_type (*standard)[MOVES_NB_COLS] = piece.is_piece_promoted ? piece_moves_promoted[piece.id] : piece_moves[piece.id];
    const move_type (*reversed)[MOVES_NB_COLS] = reversed_moves(standard);

    const move_type (*moves)[MOVES_NB_COLS];

    if (piece.player_id == player_1) {
        moves = reversed;
    } else {
        moves = standard;
    }

    return moves;
}