#include <SDL2/SDL_events.h>
#include <SDL2/SDL_mouse.h>
#include <SDL2/SDL_render.h>
#include <SDL2/SDL_stdinc.h>
#include <SDL2/SDL_video.h>
#include <SDL2/SDL_ttf.h>

#include "board.h"
#include "display.h"
#include "piece.h"
#include "input.h"
#include "menu.h"
#include "minmax.h"
#include "player.h"
#include "utils.h"

void update_display(SDL_Texture * table, SDL_Texture * background, SDL_Texture * sprites, char promote, 
                    board_t board, coord_t coord_selected, coord_t coord_hovered, board_t moves, int quality,
                    player_t * player1, player_t * player2, player_t * curr_player,
                    TTF_Font * font, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_SetRenderDrawColor(renderer, 0, 0, 0, 255);
    SDL_RenderClear(renderer);

    if (moves == NULL) {
        moves = get_blank_board();
    }

    display(table, background, sprites, 1, promote, coord_selected, coord_hovered, board, moves, quality,
            player1, player2, curr_player, font, window, renderer);

    SDL_Delay(5);

    SDL_RenderPresent(renderer);
}

void update_move(SDL_Texture * table, SDL_Texture * background, SDL_Texture * sprites, char promote,
                 board_t board, coord_t coord1, coord_t coord2, coord_t coord_hovered, board_t moves, int quality,
                 player_t * player1, player_t * player2, player_t * curr_player,
                 TTF_Font * font, SDL_Window * window, SDL_Renderer * renderer) {
    if (coord1.x != -1) {
        piece_t piece = get_piece_from_coords(board, coord1);

        // On retire la pièce avant l'animation
        board[coord1.x][coord1.y].id = blank;
        board[coord1.x][coord1.y].player_id = player_nil;
        board[coord1.x][coord1.y].is_piece_promoted = 0; 

        for (int i = 0; i < ANIM_FRAMES; ++i) {
            SDL_SetRenderDrawColor(renderer, 0, 0, 0, 255);
            SDL_RenderClear(renderer);

            display(table, background, sprites, 1, promote, coord1, coord_hovered, board, moves, quality,
                    player1, player2, curr_player, font, window, renderer);
            move_piece(i, sprites, piece, coord1, coord2, window, renderer);

            SDL_RenderPresent(renderer);
            SDL_Delay(5);
        }
    }
}

void reset_coord(coord_t * coord1, coord_t * coord2) {
    coord1->x = -1;
    coord1->y = -1;
    coord2->x = -1;
    coord2->y = -1;
}

/**
 * @brief Point d'entrée principal de l'application.
 * 
 * @param argc Nombre d'arguments de la ligne de commande.
 * @param argv Tableau des arguments de la ligne de commande.
 * @return int Code de retour du programme.
 */
int main(int argc, char ** argv) {
    (void) argc;
    (void) argv;

    SDL_Window * window = NULL;
    SDL_Renderer * renderer = NULL;

    SDL_DisplayMode screen;

    // Initialisation de la SDL & Gestion des échecs

    if (SDL_Init(SDL_INIT_VIDEO) != 0) {
        exit_sdl(0, "Error when initializing SDL", window, renderer);
    }

    // Récupération des paramètres de l'écran

    SDL_GetCurrentDisplayMode(0, &screen);

    // Création de la fenêtre

    window = SDL_CreateWindow(
        "Mini Shōgi",           // Nom de la fenêtre
        SDL_WINDOWPOS_CENTERED,     // Position x
        SDL_WINDOWPOS_CENTERED,     // Position y
        screen.w * 0.66,            // Largeur de la fenêtre
        screen.h * 0.66,            // Hauteur de la fenêtre
        SDL_WINDOW_RESIZABLE
    );

    if (window == NULL) {
        exit_sdl(0, "Error when creating the window", window, renderer);
    }

    // Création du renderer

    renderer = SDL_CreateRenderer(window, -1, SDL_RENDERER_ACCELERATED | SDL_RENDERER_PRESENTVSYNC);

    if (renderer == NULL) {
        exit_sdl(0, "Error when creating the renderer", window, renderer);
    }

    // Création de TTF : Affichage du texte

    if (TTF_Init() < 0) {
        exit_sdl(0, "Couldn't initialize SDL TTF", window, renderer);
    }

    // Chargement des textures

    SDL_Texture * table = get_texture("../img/table2.jpg", window, renderer);
    SDL_Texture * background = get_texture("../img/wooden_background.png", window, renderer);
    SDL_Texture * sprites = get_texture("../img/pieces.png", window, renderer);

    // Importation de la police

    TTF_Font * font = NULL;
    font = TTF_OpenFont("../fonts/Dosis-Bold.ttf", 65);  // La police à charger, la taille désirée
    if (font == NULL) {
        exit_sdl(0, "Can't load font", window, renderer);
    } 

    TTF_SetFontStyle(font, TTF_STYLE_BOLD);

    // Affichage du menu + Gestion des événements

    SDL_bool menu_on = SDL_TRUE;
    SDL_bool program_on = SDL_FALSE;
    SDL_bool update = SDL_TRUE;
    SDL_Event event;

    int old_x = 0, old_y = 0, x = 0, y = 0;

    char opponent_player = -1; // On choisit une exécution du programme
    int difficulty = -1; // On choisit la difficulté de l'ordi

    while (menu_on) {
        if (SDL_PollEvent(&event)) {
            switch (event.type) {
                case SDL_QUIT:
                    menu_on = SDL_FALSE;
                    program_on = SDL_FALSE;
                    break;

                case SDL_MOUSEMOTION:
                    old_x = x;
                    old_y = y;

                    x = event.button.x;
                    y = event.button.y;

                    // Rajouter une condition pour éviter de surcharger la mémoire

                    if (is_new_button_hovered(2, old_x, old_y, x, y, window)) {
                        update = SDL_TRUE;
                    }

                    break;

                case SDL_MOUSEBUTTONUP:
                    switch (event.button.button) {
                        case SDL_BUTTON_LEFT:
                            opponent_player = get_button_value(2, event.button.x, event.button.y, window);

                            if (opponent_player != -1) {
                                menu_on = SDL_FALSE;

                                if (opponent_player) {
                                    program_on = SDL_TRUE;
                                } else {
                                    program_on = SDL_FALSE;
                                }

                                update = SDL_TRUE;
                            }

                            break;

                        default:
                            break;
                    }
                    break;

                case SDL_WINDOWEVENT:
                    if (event.window.event == SDL_WINDOWEVENT_RESIZED) {
                        update = SDL_TRUE;
                    }
                    break;
                
                default:
                    break;
            }
        }

        if (update && !program_on) {
            display_menu(background, x, y, font, window, renderer);
            update = SDL_FALSE;
        }
    }

    menu_on = SDL_TRUE;
    update = SDL_TRUE;

    while ((!opponent_player) && (difficulty == -1) && (menu_on)) {
        if (SDL_PollEvent(&event)) {
            switch (event.type) {
                case SDL_QUIT:
                    menu_on = SDL_FALSE;
                    program_on = SDL_FALSE;
                    break;

                case SDL_MOUSEMOTION:
                    old_x = x;
                    old_y = y;

                    x = event.button.x;
                    y = event.button.y;

                    // Rajouter une condition pour éviter de surcharger la mémoire

                    if (is_new_button_hovered(4, old_x, old_y, x, y, window)) {
                        update = SDL_TRUE;
                    }

                    break;

                case SDL_MOUSEBUTTONUP:
                    switch (event.button.button) {
                        case SDL_BUTTON_LEFT:
                            difficulty = get_button_value(4, event.button.x, event.button.y, window);

                            if (difficulty != -1) {
                                menu_on = SDL_FALSE;
                                program_on = SDL_TRUE;
                                update = SDL_TRUE;
                            }

                            break;

                        default:
                            break;
                    }
                    break;

                case SDL_WINDOWEVENT:
                    if (event.window.event == SDL_WINDOWEVENT_RESIZED) {
                        update = SDL_TRUE;
                    }
                    break;
                
                default:
                    break;
            }
        }

        if (update && !program_on) {
            display_menu_difficulty(background, x, y, font, window, renderer);
            update = SDL_FALSE;
        }
    }

    // Choix de la difficulté 

    // Initialisation des variables utiles pour les déplacements

    coord_t coord1 = {-1, -1}, coord2 = {-1, -1}, coord_hovered = {-1, -1}, prev_coord_hovered = {-1, -1};

    board_t board = get_start_board();
    board_t new_board;

    board_t moves = get_blank_board();

    state_t * min_max_state = NULL;

    player_t * player1, * player2;

    player1 = create_player();
    player2 = create_player();

    player1->id = 0;
    player2->id = 1;

    player_t * curr_player = player1;

    int quality = 0;

    char promote = 0;
    char promote_choice = -1;
    char curr_player_winner = 0;

    // Affichage du jeu + Gestion des événements

    while (program_on) {
        if (SDL_PollEvent(&event)) {
            switch (event.type) {
                case SDL_QUIT:
                    program_on = SDL_FALSE;
                    break;

                case SDL_MOUSEMOTION:
                    prev_coord_hovered = coord_hovered;
                    coord_hovered = get_tile_coord(curr_player->id, event.button.x, event.button.y, window);

                    if (prev_coord_hovered.x != coord_hovered.x || prev_coord_hovered.y != coord_hovered.y) {
                        update = SDL_TRUE;
                    }
                    
                    break;
                
                case SDL_MOUSEBUTTONUP:
                    switch (event.button.button) {
                        case SDL_BUTTON_LEFT:
                            if ((curr_player->id == player1->id) || opponent_player) {
                                if (promote) {
                                    
                                    promote_choice = get_promote_choice(curr_player->id, coord2, event.button.x, event.button.y, window);

                                } else {

                                    if (coord1.x == -1 && coord1.y == -1) {
                                        coord1 = get_tile_coord(curr_player->id, event.button.x, event.button.y, window);

                                        if (coord1.x != -1 || coord1.y != -1) { // Si la source est valide, on affiche les coups possibles
                                            if (moves != NULL) {
                                                free_board(moves);
                                            }
                                            
                                            moves = get_all_possible_piece_slot(*curr_player, board, coord1);

                                            if (moves == NULL) { // Le coup n'est pas valide
                                                reset_coord(&coord1, &coord2);
                                            }
                                        }

                                    } else {
                                        coord2 = get_tile_coord(curr_player->id, event.button.x, event.button.y, window);

                                        if (coord2.x != -1 && coord2.y != -1) {
                                            new_board = take_move(moves, board, curr_player, coord1, coord2, &promote, &curr_player_winner);

                                            if (new_board != NULL) {

                                                update_move(table, background, sprites, promote, board, coord1, coord2, coord_hovered, moves, quality,
                                                            player1, player2, curr_player, font, window, renderer);

                                                if (*board != *new_board) {
                                                    free_board(board);
                                                }

                                                board = new_board;
                                                new_board = NULL;

                                                if (!promote && !curr_player_winner) {
                                                    if (curr_player == player1) {
                                                        curr_player = player2;
                                                    } else {
                                                        curr_player = player1;
                                                    }
                                                }

                                            }
                                        }

                                        if (!promote) {
                                            reset_coord(&coord1, &coord2);
                                        }

                                        free_board(moves);
                                        moves = get_blank_board();

                                        quality = get_board_quality(*player1, *player2, board);

                                        update = SDL_TRUE;
                                    }

                                    update = SDL_TRUE;

                                }
                            }

                            break;
                        
                        case SDL_BUTTON_RIGHT:
                            reset_coord(&coord1, &coord2);

                            free_board(moves);
                            moves = get_blank_board();

                            update = SDL_TRUE;
                            break;

                        default:
                            break;
                    }
                    break;

                case SDL_WINDOWEVENT:
                    if (event.window.event == SDL_WINDOWEVENT_RESIZED) {
                        update = SDL_TRUE;
                    }
                    break;
                
                default:
                    break;
            }
        }

        if (!opponent_player && (curr_player->id == player2->id)) {
            // Coup joué par l'algorithme min-max

            min_max_state = get_minmax_move(board, player2, player1, difficulty+1, &coord1, &coord2);

            printf("> %d %d %d %d\n", coord1.x, coord1.y, coord2.x, coord2.y);

            update_move(table, background, sprites, promote, board, coord1, coord2, coord_hovered, moves, quality,
                        player1, player2, curr_player, font, window, renderer);

            free_board(board);
            board = min_max_state->board;

            quality = -min_max_state->quality;

            curr_player_winner = is_game_over(board);

            if (!curr_player_winner) {
                curr_player = player1;
            }

            reset_coord(&coord1, &coord2);

            update = SDL_TRUE;
        }

        if (promote) {
            if (update) {
                SDL_SetRenderDrawColor(renderer, 0, 0, 0, 255);
                SDL_RenderClear(renderer);

                free_board(moves);
                moves = get_blank_board();

                display(table, background, sprites, 1, promote, coord2, coord_hovered, board, moves, quality, player1, player2, curr_player, font, window, renderer);
                ask_promote(font, window, renderer);

                SDL_RenderPresent(renderer);

                update = SDL_FALSE;
            }

            if (promote_choice != -1) {
                if (promote_choice == 1) {
                    promote_piece(&(board[coord2.x][coord2.y]));
                }

                reset_coord(&coord1, &coord2);

                if (!curr_player_winner) {
                    if (curr_player == player1) {
                        curr_player = player2;
                    } else {
                        curr_player = player1;
                    }
                }

                promote = 0;
                promote_choice = -1;
                update = SDL_TRUE;
            }
            
        }

        if (update) {
            update_display(table, background, sprites, promote, board, coord1, coord_hovered, moves, quality, player1, player2, curr_player, font, window, renderer);
            update = SDL_FALSE;
        }

        if (curr_player_winner) {
            update = SDL_TRUE;
            program_on = SDL_FALSE;
        }
    }

    // Ecran de victoire

    while (curr_player_winner) {
        if (SDL_PollEvent(&event)) {
            switch (event.type) {
                case SDL_QUIT:
                    curr_player_winner = 0;
                    break;
                
                case SDL_MOUSEBUTTONUP:
                    switch (event.button.button) {
                        case SDL_BUTTON_LEFT:
                            curr_player_winner = 0;
                            break;

                        default:
                            break;
                    }
                    break;

                case SDL_WINDOWEVENT:
                    if (event.window.event == SDL_WINDOWEVENT_RESIZED) {
                        update = SDL_TRUE;
                    }
                    break;
                
                default:
                    break;
            }
        }

        if (update) {
            display_victory_menu(background, curr_player->id, window, renderer);
            update = SDL_FALSE;
        }
    }    

    // Fermeture de la SDL

    TTF_CloseFont(font);

    SDL_DestroyTexture(sprites);
    SDL_DestroyTexture(background);
    SDL_DestroyTexture(table);

    IMG_Quit();
    TTF_Quit();

    exit_sdl(1, "Normal ending", window, renderer);

    // Libération des variables

    free(player1);
    free(player2);

    free_board(moves);
    free_board(board);

    return EXIT_SUCCESS;
}
