#include <SDL2/SDL_stdinc.h>
#include <stdio.h>
#include <stdlib.h>
#include "list.h"
#include "board.h"
#include "piece.h"

/**
 * @brief Vérifie si la liste n'est pas allouée.
 * 
 * @param list Pointeur vers la liste.
 * @return char Retourne 1 si la liste n'est pas allouée, sinon 0.
 */
char is_list_null(list_t * list) {
    if (list == NULL) {
        fprintf(stderr, "Liste nulle");
    }
    return (list == NULL);
}

/**
 * @brief Vérifie si le nœud n'est pas alloué.
 * 
 * @param node Pointeur vers le nœud.
 * @return char Retourne 1 si le nœud n'est pas alloué, sinon 0.
 */
char is_node_null(node_t * node) {
    if (node == NULL) {
        fprintf(stderr, "Node non alloué");
    }
    return (node == NULL);
}

/**
 * @brief Vérifie si la liste est vide.
 * 
 * @param list Pointeur vers la liste.
 * @return char Retourne 1 si la liste est vide, sinon 0.
 */
char is_empty_list(list_t * list) {
    if (!is_list_null(list)) {
        return (list->length == 0);
    } else {
        return 0;
    }
}

/**
 * @brief Crée et initialise une nouvelle liste.
 * 
 * @return list_t* Pointeur vers la nouvelle liste.
 */
list_t * create_list() {
    list_t * list = malloc(sizeof(list_t));
    if (!is_list_null(list)) {
        list->length = 0;
        list->head = NULL;
        list->end = NULL;
    }
    return list;
}

/**
 * @brief Crée et initialise un nouveau nœud avec le plateau de jeu.
 * 
 * @param board Pointeur vers le plateau de jeu.
 * @return node_t* Pointeur vers le nouveau nœud.
 */
node_t * create_node(board_t board) {
    node_t * node = malloc(sizeof(node_t));
    node->board = board;
    node->next = NULL;
    return node;
}

/**
 * @brief Fusionne deux listes.
 * 
 * @param list_1 Pointeur vers la première liste.
 * @param list_2 Pointeur vers la deuxième liste.
 * @return list_t* Pointeur vers la liste fusionnée.
 */
list_t * list_merge(list_t * list_1, list_t * list_2) {
    if ((is_list_null(list_1)) || (is_list_null(list_2))) {
        fprintf(stderr, "Tentative de merge d'une liste non allouée");
        return NULL;
    } else {
        if (is_empty_list(list_1)) {
            return list_2;
        } else {
            if (is_empty_list(list_2)) {
                return list_1;
            } else {
                list_1->length = list_1->length + list_2->length;
                list_1->end->next = list_2->head;
                list_1->end = list_2->end;
                free(list_2);
                return list_1;
            }
        }
    }
}

/**
 * @brief Ajoute un plateau de jeu à une liste.
 * 
 * @param list Pointeur vers la liste.
 * @param board Pointeur vers le plateau de jeu.
 * @return list_t* Pointeur vers la liste mise à jour.
 */
list_t * list_add(list_t * list, board_t board) {
    list_t * new_list;
    node_t * new_node;
    new_list = create_list();
    new_node = create_node(board);
    if (!(is_list_null(new_list)) && !(is_node_null(new_node))) {
        new_node->board = board;
        new_node->next = NULL;
        new_list->end = new_node;
        new_list->head = new_node;
        new_list->length = 1;
        list = list_merge(list, new_list);
    }
    return list;
}

/**
 * @brief Libère la mémoire allouée pour la liste et ses nœuds.
 * 
 * @param list Pointeur vers la liste.
 */
void free_list(list_t * list) {
    node_t * rm_node;
    node_t * nxt_node = list->head;
    while (nxt_node != NULL) {
        rm_node = nxt_node;
        nxt_node = nxt_node->next;
        free(rm_node);
    }
}

//Liste de coups

list_t * possible_boards_from_piece(board_t board, player_t player, coord_t coord){
    list_t * next_moves = create_list();
    board_t highlight_board = get_all_possible_piece_slot(player,board,coord);
    board_t new_result_board;
    board_t new_result_board_promoted;
    coord_t new_coord_of_piece;
    piece_t moved_piece;
    for (int i=0;i<BOARD_SIZE;i++){
        for (int j=0;j<BOARD_SIZE;j++){
            if (highlight_board[i][j].player_id == player.id){
                new_coord_of_piece.x = i;
                new_coord_of_piece.y = j;
                if (coord.x != -1) { // Si on move 
                    new_result_board = partial_copy_board(board,coord);
                } else { // Si on drop
                    new_result_board = copy_board(board);
                }
                moved_piece = highlight_board[new_coord_of_piece.x][new_coord_of_piece.y];
                new_result_board[new_coord_of_piece.x][new_coord_of_piece.y] = moved_piece;

                next_moves = list_add(next_moves,new_result_board);

                // Vérification si une promotion à lieu
                int farest_raw = (player.id == player_0)?0:BOARD_SIZE-1;
                if ((new_coord_of_piece.x == farest_raw) && is_able_to_promote(moved_piece)) {
                    new_result_board_promoted = copy_board(new_result_board);
                    new_result_board_promoted[new_coord_of_piece.x][new_coord_of_piece.y].is_piece_promoted = 1;
                    list_add(next_moves,new_result_board_promoted);
                }
                
            }
        }
    }
    return next_moves;
}

// Pour minmax
list_t * possible_moves(board_t board,player_t player){
    list_t * all_possible_boards = create_list();
    list_t * possible_boards_for_next_piece;
    coord_t coord;

    // Tous les déplacements de pièces
    for (int i=0;i<BOARD_SIZE;i++){
        for (int j=0;j<BOARD_SIZE;j++){
            if (board[i][j].player_id == player.id){
                coord.x = i;
                coord.y = j;
                possible_boards_for_next_piece = possible_boards_from_piece(board,player,coord);
                all_possible_boards = list_merge(all_possible_boards,possible_boards_for_next_piece);
            }
        }
    }

    // Tous les drops
    coord.x = -1;
    for (int i = 0 ; i < MAX_STASH_PIECE; i++) {
        if (player.available_drop_piece[i].id == blank) {
            break;
        }

        coord.y = i;
        possible_boards_for_next_piece = possible_boards_from_piece(board,player,coord);
        all_possible_boards = list_merge(all_possible_boards,possible_boards_for_next_piece);

    }

    return all_possible_boards;
}