#include <SDL2/SDL.h>
#include <SDL2/SDL_image.h>
#include <SDL2/SDL_ttf.h>
#include <SDL2/SDL_log.h>
#include <SDL2/SDL_render.h>
#include <SDL2/SDL_video.h>

#include "display.h"
#include "board.h"
#include "utils.h"

#define MAX_QUALITY_DISPLAYED 50

SDL_Texture * get_texture(char * file_image_name, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Surface * image = NULL;
    SDL_Texture * texture = NULL;

    image = IMG_Load(file_image_name);

    if (image == NULL) {
        exit_sdl(0, "Error when loading the image", window, renderer);
    }

    texture = SDL_CreateTextureFromSurface(renderer, image);
    SDL_FreeSurface(image);

    if (texture == NULL) {
        exit_sdl(0, "Surface to texture transformation failed", window, renderer);
    }

    return texture;
}

SDL_Texture * get_text_texture(char * text, TTF_Font * font, SDL_Color color, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Surface * text_surface = NULL;
    SDL_Texture * text_texture = NULL;

    // Joueur du haut

    text_surface = TTF_RenderText_Blended(font, text, color); // création du texte dans la surface 

    if (text_surface == NULL) {
        exit_sdl(0, "Can't create text surface", window, renderer);
    }

    text_texture = SDL_CreateTextureFromSurface(renderer, text_surface);

    if (text_texture == NULL) {
        exit_sdl(0, "Can't create texture from surface", window, renderer);
    }

    SDL_FreeSurface(text_surface);

    return text_texture;
}

int piece_to_index(piece_t piece) {
    int index = -1;

    index += piece.id; // Identifiant de la pièce
    index += 2 * piece.player_id * SPRITE_W; // Joueur
    index += piece.is_piece_promoted * SPRITE_W; // Promotion

    return index;
}

void highlight_tile(int r, int g, int b, int a, SDL_Rect rectangle, SDL_Renderer * renderer) {
    SDL_SetRenderDrawColor(renderer, r, g, b, a);
    SDL_RenderFillRect(renderer, &rectangle);
}

void display_piece(SDL_Texture * texture, piece_t piece, float i, float j, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect
        source = {0},               // Rectangle définissant la zone de la texture à récupérer
        window_dimensions = {0},    // Rectangle définissant la fenêtre
        destination = {0},          // Rectangle définissant où la zone source doit être déposée dans le renderer
        state = {0};                // Rectangle définissant la région correspondant à la pièce

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h); // Dimensions de la fenêtre
    SDL_QueryTexture(texture, NULL, NULL, &source.w, &source.h);           // Dimensions de l'image

    // On récupère la bonne pièce

    int index = piece_to_index(piece);

    state.w = source.w / SPRITE_W; // On découpe notre planche de sprites
    state.h = source.h / SPRITE_H;

    state.x = source.x + state.w * (index % SPRITE_W); // On sélectionne le bon sprite
    state.y = source.y + state.h * (index / SPRITE_W);

    // On fixe les dimensions de la destination

    int length = 0.8 * window_dimensions.h;         // Taille du plateau
    float tile_size = (float) length / BOARD_SIZE;  // Taille d'une case

    float zoom = 0.85 * tile_size / state.h; // Zoom de la pièce

    int x_mid = window_dimensions.w / 2;
    int y_mid = window_dimensions.h / 2;

    destination.w = state.w * zoom;
    destination.h = state.h * zoom;

    destination.x = x_mid - (length / 2.0) + (tile_size - destination.w) / 2 + j * tile_size;
    destination.y = y_mid - (length / 2.0) + (tile_size - destination.h) / 2 + i * tile_size;

    // On affiche ensuite la texture

    SDL_RenderCopy(renderer, texture, &state, &destination);   
}

void display_board(board_t board, board_t moves, char checkerboard, char promote, coord_t coord_selected, coord_t coord_hovered, char curr_player,
                   SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect window_dimensions = {0};

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h);

    // On récupère la longueur d'une ligne et le milieu de la fenêtre

    int length = 0.8 * window_dimensions.h;         // Taille du plateau
    float tile_size = (float) length / BOARD_SIZE;  // Taille d'une case

    int x_mid = window_dimensions.w / 2;
    int y_mid = window_dimensions.h / 2;

    int i, j;

    SDL_Rect rectangle;

    for (i = 0; i < BOARD_SIZE; ++i) {
        for (j = 0; j < BOARD_SIZE; ++j) {
            rectangle.w = tile_size;
            rectangle.h = tile_size;
            rectangle.x = x_mid - length / 2.0 + j * tile_size;
            rectangle.y = y_mid - length / 2.0 + i * tile_size;

            // Coloriage des cases

            if (i == coord_selected.x && j == coord_selected.y) {
                if (promote) {
                    highlight_tile(255, 192, 0, 128, rectangle, renderer);
                } else {
                    highlight_tile(255, 255, 255, 128, rectangle, renderer);
                }
            } else if (moves[i][j].id != blank) {
                highlight_tile(0, 128, 0, 128, rectangle, renderer);
            } else if (checkerboard && (i + j) % 2 == 0) {
                highlight_tile(0, 0, 0, 64, rectangle, renderer);
            }

            if (i == coord_hovered.x && j == coord_hovered.y && board[i][j].player_id == curr_player) {
                highlight_tile(255, 255, 255, 64, rectangle, renderer);
            }

            // Contour des cases

            SDL_SetRenderDrawColor(renderer, 255, 255, 255, 255);
            SDL_RenderDrawRect(renderer, &rectangle);
        }
    }
}

void display_captured_piece(SDL_Texture * texture, piece_t piece, char nb_list, int i, player_t * curr_player,
                            coord_t coord_selected, coord_t coord_hovered,
                            SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect
        source = {0},               // Rectangle définissant la zone de la texture à récupérer
        window_dimensions = {0},    // Rectangle définissant la fenêtre
        destination = {0},          // Rectangle définissant où la zone source doit être déposée dans le renderer
        state = {0};                // Rectangle définissant la région correspondant à la pièce

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h); // Dimensions de la fenêtre
    SDL_QueryTexture(texture, NULL, NULL, &source.w, &source.h);           // Dimensions de l'image

    // On récupère la bonne pièce

    int index = piece_to_index(piece);

    state.w = source.w / SPRITE_W; // On découpe notre planche de sprites
    state.h = source.h / SPRITE_H;

    state.x = source.x + state.w * (index % SPRITE_W); // On sélectionne le bon sprite
    state.y = source.y + state.h * (index / SPRITE_W);

    // On fixe les dimensions de la destination

    int length = 0.8 * window_dimensions.h; // Taille du plateau
    int zone_w = (window_dimensions.w - 1.1 * length) / 2 - length / 16.0;
    int zone_h = (window_dimensions.h) / 2 - length / 16;

    // On détermine le nombre de lignes et de colonnes en fonction de la forme de la zone de stockage

    int lines, columns;

    if (zone_w > 2 * zone_h) { // Format "paysage" : plus large que haut
        lines = 2;
        columns = 5;
    } else { // Format "portrait" : plus haut que large
        lines = 5;
        columns = 2;
    }

    // On détermine ensuite la taille d'une cellule

    int cell_w, cell_h;

    cell_w = zone_w / columns;
    cell_h = zone_h / lines;

    // On ajuste le zoom de la pièce

    float zoom = (float) cell_h / state.h;

    // On la place enfin dans la zone

    destination.w = state.w * zoom;
    destination.h = state.h * zoom;

    destination.x = length / 32 + (i % columns) * cell_w + (cell_w - destination.w) / 2;
    destination.y = length / 32 + (i / columns) * cell_h + (cell_h - destination.h) / 2;

    if (nb_list == 0) { // Liste du joueur
        destination.y += window_dimensions.h / 2;
    }

    if ((coord_selected.x == -1 && coord_selected.y == i) && (curr_player->id == nb_list &&
        (curr_player->available_drop_piece[i].id != blank))) {
        highlight_tile(255, 255, 255, 128, destination, renderer);
    } else if ((coord_hovered.x == -1 && coord_hovered.y == i) && (curr_player->id == nb_list) &&
               (curr_player->available_drop_piece[i].id != blank)) {
        highlight_tile(255, 255, 255, 64, destination, renderer);
    }

    SDL_RenderCopy(renderer, texture, &state, &destination);    
}

void display_captured(SDL_Texture * texture, piece_t list1[], piece_t list2[], coord_t coord_selected, coord_t coord_hovered, player_t * curr_player,
                      SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect
        window_dimensions = {0},    // Rectangle définissant la fenêtre
        destination = {0};          // Rectangle définissant où la zone source doit être déposée dans le renderer

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h); // Dimensions de la fenêtre

    // Affichage des rectangles

    int length = 0.8 * window_dimensions.h; // Taille du plateau

    destination.w = (window_dimensions.w - 1.1 * length) / 2 - length / 16.0;
    destination.h = window_dimensions.h / 2 - length / 16;

    SDL_SetRenderDrawColor(renderer, 255, 255, 255, 64);

    // Rectangle du haut

    destination.x = length / 32;
    destination.y = length / 32;

    SDL_RenderFillRect(renderer, &destination);

    // Rectangle du bas

    destination.x = length / 32;
    destination.y = (window_dimensions.h / 2) + (length / 32); 

    SDL_RenderFillRect(renderer, &destination);

    // Affichage des pièces

    for (int i = 0; i <10; ++i) {
        display_captured_piece(texture, list1[i], 0, i, curr_player, coord_selected, coord_hovered, window, renderer);
        display_captured_piece(texture, list2[i], 1, i, curr_player, coord_selected, coord_hovered, window, renderer);
    }
}

void display_background(SDL_Texture * table, SDL_Texture * background, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect
        source = {0},               // Rectangle définissant la zone de la texture à récupérer
        window_dimensions = {0},    // Rectangle définissant la fenêtre
        destination = {0};          // Rectangle définissant où la zone source doit être déposée dans le renderer

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h); // Dimensions de la fenêtre

    // Affichage du fond

    SDL_QueryTexture(table, NULL, NULL, &source.w, &source.h);        // Dimensions de l'image

    destination = window_dimensions;

    SDL_RenderCopy(renderer, table, &source, &destination);

    // Affichage de l'ombre

    int length = 0.8 * window_dimensions.h; // Taille du plateau

    destination.w = 1.1 * length;
    destination.h = 1.1 * length;

    int x_mid = window_dimensions.w / 2; // Point du milieu
    int y_mid = window_dimensions.h / 2;

    destination.x = x_mid - destination.w / 2 + destination.w / 32;
    destination.y = y_mid - destination.h / 2 + destination.h / 32;

    SDL_SetRenderDrawColor(renderer, 0, 0, 0, 64);
    SDL_RenderFillRect(renderer, &destination);

    // Affichage du plateau

    SDL_QueryTexture(background, NULL, NULL, &source.w, &source.h);

    destination.x -= destination.w / 32;
    destination.y -= destination.h / 32;

    SDL_RenderCopy(renderer, background, &source, &destination);
}

void display_current_player(char curr_player, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect
        window_dimensions = {0},    // Rectangle définissant la fenêtre
        destination = {0};          // Rectangle définissant où la zone source doit être déposée dans le renderer

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h); // Dimensions de la fenêtre

    // Affichage des rectangles

    int length = 0.8 * window_dimensions.h; // Taille du plateau

    destination.w = 1.1 * length;
    destination.h = length / 32;

    // Rectangle du haut

    destination.x = (window_dimensions.w - 1.1 * length) / 2;
    destination.y = (window_dimensions.h - 1.2 * length) / 2;

    if (curr_player == 1) {
        SDL_SetRenderDrawColor(renderer, 0, 255, 0, 128);
    } else {
        SDL_SetRenderDrawColor(renderer, 255, 255, 255, 64);
    }

    SDL_RenderFillRect(renderer, &destination);

    // Rectangle du bas

    destination.y = 4 * destination.y + 1.1 * length;

    if (curr_player == 0) {
        SDL_SetRenderDrawColor(renderer, 0, 255, 0, 128);
    } else {
        SDL_SetRenderDrawColor(renderer, 255, 255, 255, 64);
    }

    SDL_RenderFillRect(renderer, &destination);
}

void display_quality_bar(int quality, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect
        window_dimensions = {0},    // Rectangle définissant la fenêtre
        pos_bar = {0};          // Rectangle définissant où la barre doit être déposée dans le renderer

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h); // Dimensions de la fenêtre

    // Affichage du rectangle de progression

    int length = 0.8 * window_dimensions.h; // Taille du plateau

    int bar_length = 1.2 * length;

    // Partie haute

    pos_bar.w = length / 16;

    if (quality <= MAX_QUALITY_DISPLAYED) {
        pos_bar.h = bar_length / 2 - quality * (bar_length / 2) / MAX_QUALITY_DISPLAYED;
    } else {
        pos_bar.h = 0;
    }

    pos_bar.x = bar_length + (window_dimensions.w - bar_length) / 2;
    pos_bar.y = (window_dimensions.h - bar_length) / 2;

    SDL_SetRenderDrawColor(renderer, 0, 0, 0, 192);
    SDL_RenderFillRect(renderer, &pos_bar);   

    // Partie basse

    pos_bar.y += pos_bar.h; // On trouve le bon endroit où placer la barre

    pos_bar.h = bar_length - pos_bar.h;

    SDL_SetRenderDrawColor(renderer, 255, 255, 255, 192);
    SDL_RenderFillRect(renderer, &pos_bar);  
}

void display_quality_values(int quality, TTF_Font * font, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect
        window_dimensions = {0},    // Rectangle définissant la fenêtre
        pos_bar = {0},          // Rectangle définissant où la barre doit être déposée dans le renderer
        pos_txt = {0};          // Rectangle définissant où le texte doit être déposé dans le renderer

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h); // Dimensions de la fenêtre

    // Rappel des coordonnées de la barre

    int length = 0.8 * window_dimensions.h; // Taille du plateau
    int bar_length = 1.2 * length;

    pos_bar.w = length / 16;
    pos_bar.h = bar_length;

    pos_bar.x = bar_length + (window_dimensions.w - bar_length) / 2;
    pos_bar.y = (window_dimensions.h - bar_length) / 2;

    // Importation de la police

    SDL_Color color = {255, 255, 255, 255}; // Couleur du texte

    char text_quality[8];

    SDL_Texture * text_texture = NULL;

    // Affichage du texte

    sprintf(text_quality, "%d", quality);

    text_texture = get_text_texture(text_quality, font, color, window, renderer);

    SDL_QueryTexture(text_texture, NULL, NULL, &pos_txt.w, &pos_txt.h);

    pos_txt.x = pos_bar.x + 2 * pos_bar.w;
    pos_txt.y = pos_bar.h / 2 - pos_txt.h / 2;

    SDL_RenderCopy(renderer, text_texture, NULL, &pos_txt);

    SDL_DestroyTexture(text_texture);
}

void display_quality(int quality, TTF_Font * font, SDL_Window * window, SDL_Renderer * renderer) {
    display_quality_bar(quality, window, renderer);
    display_quality_values(quality, font, window, renderer);
}

void display(SDL_Texture * table, SDL_Texture * background, SDL_Texture * sprites, char checkerboard, char promote, 
             coord_t coord_selected, coord_t coord_hovered, board_t board, board_t moves, int quality,
             player_t * player1, player_t * player2, player_t * curr_player,
             TTF_Font * font, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_RenderClear(renderer);
    SDL_SetRenderDrawBlendMode(renderer, SDL_BLENDMODE_BLEND);
    
    display_background(table, background, window, renderer);

    display_board(board, moves, checkerboard, promote, coord_selected, coord_hovered, curr_player->id, window, renderer);

    for (int i = 0; i < BOARD_SIZE; ++i) {
        for (int j = 0; j < BOARD_SIZE; ++j) {
            display_piece(sprites, board[i][j], i, j, window, renderer);
            display_piece(sprites, board[i][j], i, j, window, renderer);
        }
    }

    display_current_player(curr_player->id, window, renderer);

    display_quality(quality, font, window, renderer);

    display_captured(sprites, player1->available_drop_piece, player2->available_drop_piece, coord_selected, coord_hovered, curr_player, window, renderer);
}

void move_piece(int iter, SDL_Texture * texture, piece_t piece, coord_t coord1, coord_t coord2, SDL_Window * window, SDL_Renderer * renderer) {
    float i = coord1.x + iter * (coord2.x - coord1.x) / ANIM_FRAMES;
    float j = coord1.y + iter * (coord2.y - coord1.y) / ANIM_FRAMES;

    display_piece(texture, piece, i, j, window, renderer);
}

void ask_promote(TTF_Font * font, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect
        window_dimensions = {0},    // Rectangle définissant la fenêtre
        pos_txt = {0};          // Rectangle définissant où le texte doit être déposé dans le renderer

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h); // Dimensions de la fenêtre

    // Importation de la police

    SDL_Color color = {255, 255, 255, 255}; // Couleur du texte

    SDL_Texture * text_texture;

    // Positionnement du texte

    int length = 0.8 * window_dimensions.h; // Taille du plateau
    int bar_length = 1.2 * length;          // Taille de la barre des qualités

    pos_txt.x = bar_length + (window_dimensions.w - bar_length) / 2 + bar_length / 16;

    // La pièce peut être promue

    text_texture = get_text_texture("Une piece peut etre promue.", font, color, window, renderer);

    SDL_QueryTexture(text_texture, NULL, NULL, &pos_txt.w, &pos_txt.h);

    pos_txt.y = 0.1 * window_dimensions.h;
    SDL_RenderCopy(renderer, text_texture, NULL, &pos_txt);

    SDL_DestroyTexture(text_texture);

    // Cliquez dessus pour la promouvoir

    text_texture = get_text_texture("Cliquez dessus pour la promouvoir.", font, color, window, renderer);

    SDL_QueryTexture(text_texture, NULL, NULL, &pos_txt.w, &pos_txt.h);

    pos_txt.y += 1.15 * pos_txt.h;
    SDL_RenderCopy(renderer, text_texture, NULL, &pos_txt);

    SDL_DestroyTexture(text_texture);

    // Cliquez ailleurs pour annuler.

    text_texture = get_text_texture("Cliquez ailleurs pour ignorer.", font, color, window, renderer);

    SDL_QueryTexture(text_texture, NULL, NULL, &pos_txt.w, &pos_txt.h);

    pos_txt.y += 1.15 * pos_txt.h;
    SDL_RenderCopy(renderer, text_texture, NULL, &pos_txt);

    SDL_DestroyTexture(text_texture);
}

void exit_sdl(char ok, char const * msg, SDL_Window * window, SDL_Renderer * renderer) {
    char msg_formatted[255];
    int l;

    if (!ok) {
        strncpy(msg_formatted, msg, 250);
        l = strlen(msg_formatted);
        strcpy(msg_formatted + l, " : %s\n");
        SDL_Log(msg_formatted, SDL_GetError());
    }

    if (renderer != NULL) {
        SDL_DestroyRenderer(renderer);
        renderer = NULL;
    }

    if (window != NULL) {
        SDL_DestroyWindow(window);
        window = NULL;
    }

    SDL_Quit();

    if (!ok) {
        exit(EXIT_FAILURE);
    }
}