#include "board.h"
#include "piece.h"
#include "piece_const.h"
#include "player.h"
#include "utils.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/**
 * @brief Imprime le plateau de jeu.
 *
 * @param board Le plateau de jeu à imprimer.
 */
void print_board(board_t board){
    int i,j;
    for (i = 0; i < BOARD_SIZE; i++){
        for (j = 0; j < BOARD_SIZE; j++){
            printf(" %d %d |", board[i][j].id, board[i][j].player_id);
        }
        printf("\n");
    }
}

board_t partial_copy_board(board_t board, coord_t coord){
    int i,j;
    board_t copy_board = get_blank_board();
    for (i=0;i<BOARD_SIZE;i++){
        for (j=0;j<BOARD_SIZE;j++){
            copy_board[i][j].id = board[i][j].id;
            copy_board[i][j].is_piece_promoted = board[i][j].is_piece_promoted;
            copy_board[i][j].player_id = board[i][j].player_id;
        }
    }
    copy_board[coord.x][coord.y].id = blank;
    copy_board[coord.x][coord.y].is_piece_promoted = 0;
    copy_board[coord.x][coord.y].player_id = player_nil;
    return copy_board;
}

/**
 * @brief Permet de récupérer la pièce aux coordonnées fournies.
 *
 * @param board Le plateau de jeu.
 * @param coords Les coordonnées de la pièce à récupérer.
 * @return La pièce aux coordonnées spécifiées.
 */
piece_t get_piece_from_coords(board_t board, coord_t coords){
    int x, y;
    piece_t piece;
    x = coords.x;
    y = coords.y;
    piece = board[x][y];
    return piece;
}


void free_board(board_t board) {
    for (int i = 0; i < BOARD_SIZE; i++) {
        if (board[i] != NULL){
            free(board[i]);
        }
    }
    free(board);
}

/**
 * @brief Permet de créer un plateau complet à partir des pièces et des joueurs les contrôlant.
 *
 * @param pieces_pos Positions des pièces sur le plateau.
 * @param player_pos Positions des joueurs sur le plateau.
 * @return Un pointeur vers le plateau de jeu initialisé.
 */
board_t get_board_by_array(int pieces_pos[BOARD_SIZE][BOARD_SIZE], char player_pos[BOARD_SIZE][BOARD_SIZE]) {
    board_t board = malloc(BOARD_SIZE * sizeof(piece_t*)); // board = ** piece_t
    for (int i = 0; i < BOARD_SIZE; i++) {
        board[i] = malloc(BOARD_SIZE * sizeof(piece_t));
        for (int j = 0 ; j < BOARD_SIZE; j++) {
            piece_t piece = get_blank_piece();
            piece.id = pieces_pos[i][j];
            piece.player_id = player_pos[i][j];

            // Pour avoir le bon sprite, le roi noir doit être promu
            if ((piece.id == king) && (piece.player_id == 1)) {
                piece.is_piece_promoted = 1;
            }

            board[i][j] = piece;
        }
    }
    return board;
}

/**
 * @brief Fonction donnant le plateau de départ d'une partie de mini-shogi.
 *
 * @return Un pointeur vers le plateau de jeu initialisé pour le départ.
 */
board_t get_start_board() {
    int pieces_pos[BOARD_SIZE][BOARD_SIZE] = {
        {rook, bishop, silver_general, gold_general, king},
        {blank, blank, blank, blank, pawn},
        {blank, blank, blank, blank, blank},
        {pawn, blank, blank, blank, blank},
        {king, gold_general, silver_general, bishop, rook}
    };

    char player_pos[BOARD_SIZE][BOARD_SIZE] = {
        {1, 1, 1, 1, 1},
        {player_nil, player_nil, player_nil, player_nil, 1},
        {player_nil, player_nil, player_nil, player_nil, player_nil},
        {0, player_nil, player_nil, player_nil, player_nil},
        {0, 0, 0, 0, 0}
    };

    return get_board_by_array(pieces_pos, player_pos);
}

/**
 * @brief Crée un plateau vide.
 *
 * @return Un pointeur vers un plateau de jeu vide.
 */
board_t get_blank_board() {
    int pieces_pos[BOARD_SIZE][BOARD_SIZE] = {0};

    char player_pos[BOARD_SIZE][BOARD_SIZE] = {
        {player_nil, player_nil, player_nil, player_nil, player_nil},
        {player_nil, player_nil, player_nil, player_nil, player_nil},
        {player_nil, player_nil, player_nil, player_nil, player_nil},
        {player_nil, player_nil, player_nil, player_nil, player_nil},
        {player_nil, player_nil, player_nil, player_nil, player_nil},
    };

    return get_board_by_array(pieces_pos, player_pos);  
}

board_t get_single_piece_board(piece_t piece, player_t player) {
    int pieces_pos[BOARD_SIZE][BOARD_SIZE];
    char player_pos[BOARD_SIZE][BOARD_SIZE];

    for (int i = 0; i < BOARD_SIZE ; i++) {
        for (int j = 0 ; j < BOARD_SIZE; j++) {
            pieces_pos[i][j] = piece.id;
            player_pos[i][j] = player.id;
        }
    }

    return get_board_by_array(pieces_pos, player_pos);  
}

/**
 * @brief Evalue la qualité de la position du point de vue du joueur actif (joueur jouant le prochain coup).
 *
 * @param act_player Le joueur actif.
 * @param res_player Le joueur adverse.
 * @param board Le plateau de jeu.
 * @return La valeur de la qualité de la position.
 */
int get_board_quality(player_t act_player, player_t res_player, board_t board) {
    if (NULL == board) {
        fprintf(stderr, "Tentative d'obtenir la qualité d'un plateau NULL");
        return -1;
    }

    int value = 0;
    // Qualité sur le plateau
    for (int i = 0; i < BOARD_SIZE; i++) {
        for (int j = 0; j < BOARD_SIZE; j++) {
            piece_t piece = board[i][j];
            if (piece.player_id == act_player.id) {
                value += get_piece_value(piece) + 1; // +1 car une pièce sur le plateau vaut plus qu'une pièce en stock
            } else if (piece.player_id == res_player.id) {
                value -= get_piece_value(piece) + 1;
            }
        }
    }
    // Qualité dans les stockages
    for (int i = 0; i < MAX_STASH_PIECE; i++) {
        value += get_piece_value(act_player.available_drop_piece[i]);
        value -= get_piece_value(res_player.available_drop_piece[i]);
    }

    return value;
}

/**
 * @brief Génère tous les mouvements possibles pour une pièce donnée sur le plateau.
 *
 * @param player Le joueur actif.
 * @param board Le plateau de jeu.
 * @param coord Les coordonnées de la pièce à déplacer.
 * @return Un pointeur vers un plateau de jeu avec tous les mouvements possibles pour la pièce, ou NULL en cas d'erreur.
 */
board_t get_all_possible_move_for_piece(player_t player, board_t board, coord_t coord){
    int i, j;
    char next_move_is_possible;
    int possible_next_move_x, possible_next_move_y;
    coord_t basic_coords_player_i;
    board_t nv_coup = get_blank_board();
    piece_t piece = get_piece_from_coords(board, coord);
    if (piece.player_id != player.id){
        fprintf(stderr, "Tentative de bouger une pièce ne lui appartenant pas");
        return NULL;
    }
    if (player.id == player_0){
        basic_coords_player_i.x = 2;
        basic_coords_player_i.y = 1;
    }
    if (player.id == player_1){
        basic_coords_player_i.x = 1;
        basic_coords_player_i.y = 1;
    }
    const move_type (*moves)[3] = get_piece_moves(piece);
    for (i = 0; i < 4; i++) {
        for (j = 0; j < 3; j++) {
            possible_next_move_x = coord.x + i - basic_coords_player_i.x; //Coordonnées du prochain coup à tester
            possible_next_move_y = coord.y + j - basic_coords_player_i.y;
            if (moves[i][j] == move_one) {
                if ((possible_next_move_x < BOARD_SIZE) && (possible_next_move_y < BOARD_SIZE) && (possible_next_move_x >= 0) && (possible_next_move_y >=0)) {
                    if (board[possible_next_move_x][possible_next_move_y].player_id != board[coord.x][coord.y].player_id) {
                        nv_coup[possible_next_move_x][possible_next_move_y] = piece;
                    }
                }
            } else {
                if (moves[i][j] == move_all) {
                    next_move_is_possible = 1;
                    while ((possible_next_move_x < BOARD_SIZE) && (possible_next_move_y < BOARD_SIZE) && (possible_next_move_x >= 0) && (possible_next_move_y >=0) && (next_move_is_possible)) {
                        if (board[possible_next_move_x][possible_next_move_y].player_id != player_nil) {
                            next_move_is_possible = 0; 
                        }
                        if (board[possible_next_move_x][possible_next_move_y].player_id != player.id) { // Si la pièce appartient à l'adversaire, on peut la manger, mais pas la traverser
                            nv_coup[possible_next_move_x][possible_next_move_y] = piece; 
                        }
                        possible_next_move_x += i - basic_coords_player_i.x;
                        possible_next_move_y += j - basic_coords_player_i.y;
                    }

                }
            }
        }
    }
    return nv_coup;
}

/**
 * @brief Vérifie si un mouvement est légal.
 *
 * @param availables_moves Le plateau avec les mouvements disponibles.
 * @param src Les coordonnées de la pièce à déplacer.
 * @param dst Les coordonnées de destination.
 * @return 1 si le mouvement est légal, 0 sinon.
 */
char is_move_legal(board_t availables_moves, coord_t src, coord_t dst) {
    if (src.x == -1) {
        if ((src.y == -1) || (dst.x == -1) || (dst.y == -1)) {
            return 0;
        } else {
            return (availables_moves[dst.x][dst.y].id == blank);
        }
    }

    printf("%d %d %d %d\n", src.x, src.y, dst.x, dst.y);

    if ((src.x == dst.x) && (src.y == dst.y)) {
        return 0;
    }

    return (availables_moves[dst.x][dst.y].id != blank);
}

char is_move_taking_opponent_piece(board_t original_board, coord_t dst) {
    return (original_board[dst.x][dst.y].player_id != player_nil);
}

board_t copy_board(board_t original_board) {
    board_t new_board = get_blank_board();
    for (int i = 0; i<BOARD_SIZE; i++) {
        memcpy(new_board[i], original_board[i], BOARD_SIZE*sizeof(piece_t));
    }
    return new_board;
}

char is_cols_having_allied_pawn(board_t board, player_t player, int cols_nb) {
    for (int i = 0; i < BOARD_SIZE ; i++) {
        piece_t current_piece = board[i][cols_nb];
        if ((current_piece.id == pawn) && (current_piece.player_id == player.id)) {
            return 1;
        }
    }

    return 0;
}

board_t get_all_possible_piece_slot(player_t player, board_t board, coord_t coord) {
    // Si coordonée invalide.
    if (coord.y == -1) {
        return board;
    }
    
    if (coord.x == -1) {
        return get_all_possible_drop_for_piece(player, board, coord);
    } else {
        return get_all_possible_move_for_piece(player, board, coord);
    }
}

board_t get_all_possible_drop_for_piece(player_t player, board_t board, coord_t drop_index) { // drop_index = {-1, i}
    piece_t drop_piece = player.available_drop_piece[drop_index.y];
    board_t available_drop_bord = get_single_piece_board(drop_piece, player);

    int farest_raw = (player.id == player_0)?0:BOARD_SIZE-1;
    int second_farest_raw = (player.id == player_0)?1:BOARD_SIZE-2;

    if (drop_piece.id == blank) {
        fprintf(stderr, "Tentative de parachuter une pièce vide\n");
        return NULL;
    }

    // Si on veut drop un pion/lance/cavalier, on ne peut pas le faire sur la première ligne
    if ((drop_piece.id == pawn) || (drop_piece.id == spear) || (drop_piece.id == knight)) {
        for (int i = 0; i < BOARD_SIZE; i++) {
            available_drop_bord[farest_raw][i] = get_blank_piece();
        }

        // On ne peut pas drop sur la deuxième si c'est un cavalier
        if (drop_piece.id == knight) {
            for (int i = 0; i < BOARD_SIZE; i++) {
                available_drop_bord[second_farest_raw][i] = get_blank_piece();
            }
        }
    }

    // S'il y a un pion allié dans la colonne et qu'on veut drop un pion, on ne peut pas drop dans la colonne.
    if (drop_piece.id == pawn) {
        for (int i = 0; i < BOARD_SIZE; i++) {
            if (is_cols_having_allied_pawn(board, player, i)) {
                for (int j = 0; j < BOARD_SIZE; j++) {
                    available_drop_bord[j][i] = get_blank_piece();
                }
            }
        }
    }

    // Si la case n'est pas vide on ne peut pas drop
    for (int i = 0; i < BOARD_SIZE; i++) {
        for (int j = 0; j < BOARD_SIZE; j++) {
            if (board[i][j].id != blank) {
                available_drop_bord[i][j] = get_blank_piece();
            }
        }
    }

    return available_drop_bord;
}

board_t take_move(board_t availables_moves, board_t original_board, player_t * current_player, 
    coord_t src, coord_t dst, char * is_needed_to_ask_for_promote, char * is_current_player_winner) {

    board_t after_move_board = copy_board(original_board);
    piece_t moved_piece;

    if (-1 == src.x)  {// S'il s'agit d'un parachutage
        moved_piece = current_player->available_drop_piece[src.y];
        if (is_move_legal(original_board, src, dst)) {
            printf("bloup\n");
            remove_drop_piece_to_player(current_player, moved_piece);
        } else {
            return original_board;
        }
    } else { // Si c'est un coup "classique"
        if (!is_move_legal(availables_moves, src, dst)) { // Si le coup est illégal
            free(after_move_board);
            return NULL;

        } else { // Si le coup est légal
            // Si on mange une pièce
            moved_piece = original_board[src.x][src.y];
            if (is_move_taking_opponent_piece(original_board, dst)) {
                add_drop_piece_to_player(current_player, original_board[dst.x][dst.y]);
                if (original_board[dst.x][dst.y].id == king) {
                    *is_current_player_winner = 1;
                }
            }


            after_move_board[src.x][src.y] = get_blank_piece();
        }

        // Vérification si une promotion à lieu
        int farest_raw = (current_player->id == player_0)?0:BOARD_SIZE-1;
        if ((dst.x == farest_raw) && is_able_to_promote(moved_piece)) {
            *is_needed_to_ask_for_promote = 1;
        }
    }
    after_move_board[dst.x][dst.y] = moved_piece;

    return after_move_board;
};

int get_number_of_piece_in_board(board_t board) {
    int sum = 0;
    
    for (int i = 0 ; i < BOARD_SIZE ; i++) {
        for (int j = 0 ; j < BOARD_SIZE ; j++) {
            if (board[i][j].id != blank) {
                sum++;
            }
        }
    }

    return sum;
}

char is_game_over(board_t board){
    int i,j, k = 2; // 2 est le nombre de rois
    for (i=0;i<BOARD_SIZE;i++){
        for (j=0;j<BOARD_SIZE;j++){
            if (board[i][j].id == king){
                k -=1;
            }
        }
    }
    return (k != 0);
}

void get_eaten_piece(board_t prev_board,board_t new_board, coord_t * coord_src, coord_t * coord_dst, player_t player){
    int i,j;
    player_id id = player.id;
    for (i=0;i<BOARD_SIZE;i++){
        for (j=0;j<BOARD_SIZE;j++){
            printf("%d %d\n",prev_board[i][j].player_id,new_board[i][j].player_id);
            if ((prev_board[i][j].player_id == (1 - id)) && (new_board[i][j].player_id == id)){
                coord_dst->x = i;
                coord_dst->y = j;
                id = new_board[i][j].player_id;
            }
            if ((prev_board[i][j].player_id == id) && (new_board[i][j].player_id == player_nil)){
                coord_src->x = i;
                coord_src->y = j;
            }
        }
    }
    printf("%d %d\n",coord_dst->x,coord_dst->y);
}

coord_t get_dropped_piece(board_t prev_board,board_t new_board){ //Sur un board ayant subi un drop, renvoyez la piece
    int i,j;
    coord_t coord;
    for (i=0;i<BOARD_SIZE;i++){
        for (j=0;j<BOARD_SIZE;j++){
            if (((prev_board[i][j].player_id == player_nil) && (new_board[i][j].player_id != player_nil)) 
              ||((prev_board[i][j].player_id == player_1) && (new_board[i][j].player_id == player_0))){
                coord.x = i;
                coord.y = j;
            }
        }
    }
    return coord;
}

void get_moved_piece(board_t prev_board,board_t new_board, coord_t * coord_src, coord_t * coord_dst, player_t player){
    int i,j;
    player_id id = player.id;
    for (i=0;i<BOARD_SIZE;i++){
        for (j=0;j<BOARD_SIZE;j++){
            printf("%d %d\n",prev_board[i][j].player_id,new_board[i][j].player_id);
            if ((prev_board[i][j].player_id == player_nil) && (new_board[i][j].player_id == id)){
                coord_dst->x = i;
                coord_dst->y = j;
                id = new_board[i][j].player_id;
            }
            if ((prev_board[i][j].player_id == id) && (new_board[i][j].player_id == player_nil)){
                coord_src->x = i;
                coord_src->y = j;
            }
        }
    }
    printf("%d %d\n",coord_dst->x,coord_dst->y);
}