#include <SDL2/SDL.h>
#include <SDL2/SDL_image.h>
#include <stdio.h>

void exit_sdl(char ok, char const * msg, SDL_Window * window, SDL_Renderer * renderer) {
    char msg_formatted[255];
    int l;

    if (!ok) {
        strncpy(msg_formatted, msg, 250);
        l = strlen(msg_formatted);
        strcpy(msg_formatted + l, " : %s\n");
        SDL_Log(msg_formatted, SDL_GetError());
    }

    if (renderer != NULL) {
        SDL_DestroyRenderer(renderer);
        renderer = NULL;
    }

    if (window != NULL) {
        SDL_DestroyWindow(window);
        window = NULL;
    }

    SDL_Quit();

    if (!ok) {
        exit(EXIT_FAILURE);
    }
}

SDL_Texture * load_texture_from_image(char * file_image_name, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Surface * image = NULL;
    SDL_Texture * texture = NULL;

    image = IMG_Load(file_image_name);

    if (image == NULL) {
        exit_sdl(0, "Error when loading the image", window, renderer);
    }

    texture = SDL_CreateTextureFromSurface(renderer, image);
    SDL_FreeSurface(image);

    if (texture == NULL) {
        exit_sdl(0, "Surface to texture transformation failed", window, renderer);
    }

    return texture;
}

void play_with_background(SDL_Texture * background, int offset, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect
        source = {0},               // Rectangle définissant la zone de la texture à récupérer
        window_dimensions = {0},    // Rectangle définissant la fenêtre
        destination = {0};          // Rectangle définissant où la zone source doit être déposée dans le renderer

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h); // Dimensions de la fenêtre
    SDL_QueryTexture(background, NULL, NULL, &source.w, &source.h);           // Dimensions de l'image

    SDL_RenderClear(renderer);

    // On fixe les dimensions de la destination, puis on affiche la texture

    destination.w = window_dimensions.w;
    destination.h = window_dimensions.h;
    destination.x = (window_dimensions.x - offset) % window_dimensions.w;
    destination.y = window_dimensions.y;

    SDL_RenderCopy(renderer, background, &source, &destination);

    destination.x = window_dimensions.w + (window_dimensions.x - offset) % window_dimensions.w;

    SDL_RenderCopy(renderer, background, &source, &destination);
}

void play_with_texture(SDL_Texture * texture, float zoom, SDL_Window * window, SDL_Renderer * renderer) {
    SDL_Rect
        source = {0},               // Rectangle définissant la zone de la texture à récupérer
        window_dimensions = {0},    // Rectangle définissant la fenêtre
        destination = {0};          // Rectangle définissant où la zone source doit être déposée dans le renderer

    SDL_GetWindowSize(window, &window_dimensions.w, &window_dimensions.h); // Dimensions de la fenêtre
    SDL_QueryTexture(texture, NULL, NULL, &source.w, &source.h);           // Dimensions de l'image

    // On fixe les dimensions de la destination

    destination.w = source.w * zoom;
    destination.h = source.h * zoom;

    destination.x = (window_dimensions.w - destination.w) / 2;
    destination.y = (window_dimensions.h - destination.h) / 2;

    // On affiche ensuite la texture

    SDL_RenderCopy(renderer, texture, &source, &destination);
}

int main(int argc, char ** argv) {
    (void) argc;
    (void) argv;

    SDL_Window * window = NULL;
    SDL_Renderer * renderer = NULL;

    SDL_DisplayMode screen;

    // Initialisation de la SDL & Gestion des échecs

    if (SDL_Init(SDL_INIT_VIDEO) != 0) {
        exit_sdl(0, "Error when initializing SDL", window, renderer);
    }

    // Récupération des paramètres de l'écran

    SDL_GetCurrentDisplayMode(0, &screen);

    // Création de la fenêtre

    window = SDL_CreateWindow(
        "Animation de sprite",      // Nom de la fenêtre
        SDL_WINDOWPOS_CENTERED,     // Position x
        SDL_WINDOWPOS_CENTERED,     // Position y
        screen.w * 0.66,            // Largeur de la fenêtre
        screen.h * 0.66,            // Hauteur de la fenêtre
        SDL_WINDOW_RESIZABLE
    );

    if (window == NULL) {
        exit_sdl(0, "Error when creating the window", window, renderer);
    }

    // Création du renderer

    renderer = SDL_CreateRenderer(window, -1, SDL_RENDERER_ACCELERATED | SDL_RENDERER_PRESENTVSYNC);

    if (renderer == NULL) {
        exit_sdl(0, "Error when creating the renderer", window, renderer);
    }

    // Importation des textures

    SDL_Texture * background = load_texture_from_image("./img/backgroundColorFall.png", window, renderer);

    SDL_Texture * characters[11];
    char character_filename[255];

    int i;

    for (i = 0; i < 8; ++i) {
        sprintf(character_filename, "./img/character_femalePerson_walk%d.png", i);
        characters[i] = load_texture_from_image(character_filename, window, renderer);
    }

    for (i = 8; i < 11; ++i) {
        sprintf(character_filename, "./img/character_femalePerson_attack%d.png", i-8);
        characters[i] = load_texture_from_image(character_filename, window, renderer);
    }

    // Remplissage de la fenêtre

    SDL_bool program_on = SDL_TRUE;
    SDL_bool paused = SDL_FALSE;
    SDL_Event event;

    int offset = 0;

    i = 0;

    while (program_on) {
        if (SDL_PollEvent(&event)) {
            switch (event.type) {
                case SDL_QUIT:
                    program_on = SDL_FALSE;
                    break;

                case SDL_KEYDOWN:
                    switch (event.key.keysym.sym) {
                        case SDLK_SPACE:
                            paused = !paused;
                            break;
                        
                        default:
                            break;
                    }
                
                default:
                    break;
            }
        }

        if (!paused) {
            play_with_background(background, offset, window, renderer);
            play_with_texture(characters[i], 1, window, renderer);
            SDL_RenderPresent(renderer);
            SDL_Delay(100);
        }

        // Le personnage change d'état
        i = (i + 1) % 11;

        // Le décor bouge si le personnage n'attaque pas
        if (i < 8) {
            offset = (offset + 10) % screen.w;
        }
    }

    // Fermeture de la SDL

    SDL_DestroyTexture(background);

    for (i = 0; i < 11; ++i) {
        SDL_DestroyTexture(characters[i]);
    }

    IMG_Quit();

    exit_sdl(1, "Normal ending", window, renderer);
    return EXIT_SUCCESS;
}
