#include <SDL2/SDL.h>
#include <SDL2/SDL_events.h>
#include <SDL2/SDL_log.h>
#include <SDL2/SDL_render.h>
#include <SDL2/SDL_image.h> 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define TEXTURE_NB 10
#define PARALLAX_NB 5
#define FRAME_NB 8

typedef struct {
   int x;
   int y;
   int value;
} valpoint_t;

void end_sdl(char ok, char const * msg, SDL_Window * window, SDL_Renderer * renderer, SDL_Texture ** textures) {
    char msg_formated[255];
    int l;

    if (!ok) {
        strncpy(msg_formated, msg, 250);
        l = strlen(msg_formated);
        strcpy(msg_formated + l, " : %s\n");

        SDL_Log(msg_formated, SDL_GetError());
    }

    if (renderer != NULL) {
        SDL_DestroyRenderer(renderer);
        renderer = NULL;
    }
    if (window != NULL) {
        SDL_DestroyWindow(window);
        window = NULL;
    }

    if (textures != NULL) {
        for (int i = 0; i<TEXTURE_NB; i++) {
            if (NULL != textures[i]) {
                free(textures[i]);
            }
        }
        free(textures);

    }

    IMG_Quit();
    SDL_Quit();

    if (!ok) {
        exit(EXIT_FAILURE);
    }
}

SDL_Texture* load_texture_from_image(char * file_image_name, SDL_Window * window, SDL_Renderer * renderer, SDL_Texture ** textures){
    SDL_Texture *my_texture; 
    my_texture = IMG_LoadTexture(renderer, file_image_name);
    if (my_texture == NULL) end_sdl(0, "Echec du chargement de l'image dans la texture", window, renderer, textures);

    return my_texture;
}

void draw_texture(SDL_Renderer *renderer, SDL_Texture *my_texture, valpoint_t dim_window, int x, int y, int frame)  {
    SDL_Rect source = {0}, window_dimensions = {0};

    // Là où on place la frame
    window_dimensions.x = x;
    window_dimensions.y = y;

    // Si ça a des frames, alors c'est l'animation du personnage
    if (-1 != frame) {
        source.w = 128;
        source.h = 128;
        source.x = 128*frame;

        window_dimensions.w = 128;
        window_dimensions.h = 128;
    } else { // Sinon en plein écran
        window_dimensions.w = dim_window.x;
        window_dimensions.h = dim_window.y;
        SDL_QueryTexture(my_texture, NULL, NULL, &source.w, &source.h);
    }

    
    SDL_RenderCopy(renderer, my_texture, &source,&window_dimensions);
}

void draw(SDL_Renderer *renderer, valpoint_t dim_window, SDL_Texture ** textures, int parallax_offset1[], int parallax_offset2[], int frame) {

    // Fond
    SDL_SetRenderDrawColor(renderer, 255, 255, 255, 255); 
    SDL_RenderClear(renderer);
    
    // Effet Parallax
    for (int i = 0; i < PARALLAX_NB; i++) {
        draw_texture(renderer, textures[i], dim_window, parallax_offset1[i], 0, -1);
        draw_texture(renderer, textures[i], dim_window, parallax_offset2[i], 0, -1);
    }

    // Personnage
    draw_texture(renderer, textures[PARALLAX_NB], dim_window, dim_window.x/2 - 128/2, dim_window.y-128, frame);
}

void game_logic(valpoint_t dim_window, int parallax_offset1[], int parallax_offset2[], int * frame, int iter) {
    for (int i = 0; i<PARALLAX_NB; i++) {
        parallax_offset1[i] -= i + 1;  // Change increment to decrement
        parallax_offset2[i] -= i + 1;  // Change increment to decrement
        
        if (parallax_offset1[i] <= -dim_window.x) {  // Update condition for resetting
            parallax_offset1[i] = dim_window.x;
        }

        if (parallax_offset2[i] <= -dim_window.x) {  // Update condition for resetting
            parallax_offset2[i] = dim_window.x;
        }
    }

    if (iter == 0) {
        *frame = (*frame + 1) % FRAME_NB;
    }
}

int main(int argc, char **argv) {
    (void)argc;
    (void)argv;

    SDL_Window *window = NULL;
    SDL_Renderer *renderer = NULL;

    SDL_DisplayMode screen;

    if (SDL_Init(SDL_INIT_VIDEO) != 0) {
        end_sdl(0, "ERROR SDL INIT", window, renderer, NULL);
    }

    SDL_GetCurrentDisplayMode(0, &screen);

    valpoint_t dim_window;
    dim_window.x = screen.w * 0.66;
    dim_window.y = screen.h * 0.66;

    window = SDL_CreateWindow("City Walking Simulator 2077", SDL_WINDOWPOS_CENTERED, SDL_WINDOWPOS_CENTERED, dim_window.x, dim_window.y, SDL_WINDOW_OPENGL);
    if (window == NULL) {
        end_sdl(0, "ERROR WINDOW CREATION", window, renderer, NULL);
    }

    renderer = SDL_CreateRenderer(window, -1, SDL_RENDERER_ACCELERATED | SDL_RENDERER_PRESENTVSYNC);
    if (renderer == NULL) {
        end_sdl(0, "ERROR RENDERER CREATION", window, renderer, NULL);
    }

    // Chargement des textures
    SDL_Texture ** textures = calloc(TEXTURE_NB, sizeof(SDL_Texture*));
    for (int i = 0; i<PARALLAX_NB; i++)  {
        char img_name[20];
        sprintf(img_name, "./img/%d.png", i+1);
        textures[i] = load_texture_from_image(img_name, window, renderer, textures);
    }

    textures[PARALLAX_NB] = load_texture_from_image("./img/run.png", window, renderer, textures);

    // Initialisation du parallax
    int parallax_offset1[PARALLAX_NB] = {0};
    int parallax_offset2[PARALLAX_NB];
    int texture_width;
    SDL_QueryTexture(textures[0], NULL, NULL, &texture_width, NULL);

    for (int i = 0; i<PARALLAX_NB; i++) {
        parallax_offset2[i] = dim_window.x;
    }

    // Initialisation des frames
    int frame = 0;
    int iter = 0;

    SDL_Event event;
    char running = 1;
    while (running) {
        while (SDL_PollEvent(&event)) {
            switch (event.type) {
                case SDL_QUIT:
                    running = 0;
                    break;
            }
        }
        iter = (iter + 1)%5;
        game_logic(dim_window, parallax_offset1, parallax_offset2, &frame, iter);
        draw(renderer, dim_window, textures, parallax_offset1, parallax_offset2, frame);   
        SDL_RenderPresent(renderer);
        SDL_Delay(10);
    }

    end_sdl(1, "Normal ending", window, renderer, textures);
    return EXIT_SUCCESS;
}
