#include <SDL2/SDL.h>
#include <stdio.h>
#include <time.h>

#include <math.h>

#define PI 3.14159265358979323846

#define MAX_SNAKES 20
#define MAX_PARTS 30

struct snake_part {
    int x;
    int y;

    int length;
    double angle;

    SDL_Color color;
};

struct snake {
    struct snake_part * parts;
    
    int size;
};

void exit_sdl(char ok, char const * msg, SDL_Window * window, SDL_Renderer * renderer) {
    char msg_formatted[255];
    int l;

    if (!ok) {
        strncpy(msg_formatted, msg, 250);
        l = strlen(msg_formatted);
        strcpy(msg_formatted + l, " : %s\n");
        SDL_Log(msg_formatted, SDL_GetError());
    }

    if (renderer != NULL) {
        SDL_DestroyRenderer(renderer);
        renderer = NULL;
    }

    if (window != NULL) {
        SDL_DestroyWindow(window);
        window = NULL;
    }

    SDL_Quit();

    if (!ok) {
        exit(EXIT_FAILURE);
    }
}

void draw_snake(SDL_Renderer * renderer, struct snake * snake) {
    int i;

    for (i = 0; i < snake->size - 1; ++i) {
        struct snake_part * sp = &(snake->parts[i]);

        // Calcul des points

        snake->parts[i+1].x = sp->x + sp->length * cos(sp->angle);
        snake->parts[i+1].y = sp->y + sp->length * sin(sp->angle);

        // Affichage du rectangle

        SDL_SetRenderDrawColor(renderer, sp->color.r, sp->color.g, sp->color.b, 255);

        SDL_RenderDrawLine(renderer, snake->parts[i].x, snake->parts[i].y,
                                     snake->parts[i+1].x, snake->parts[i+1].y);

        // Le serpent s'entortille
        
        sp->angle += (rand() % 40 - 20) * PI / 180;

        // Sa couleur change

        sp->color.r += (rand() % 10 - 5);
        sp->color.g += (rand() % 10 - 5);
        sp->color.b += (rand() % 10 - 5);
    }

    // Le serpent se déplace

    int length = rand() % 3;

    snake->parts[0].x -= (length * cos(snake->parts[0].angle));
    snake->parts[0].y -= (length * sin(snake->parts[0].angle));
}

void draw_snakes(SDL_Renderer * renderer, struct snake * snakes, int nb_snakes) {
    int i;

    for (i = 0; i < nb_snakes; ++i) {
        draw_snake(renderer, &snakes[i]);
    }
}

int main(int argc, char ** argv) {
    (void) argc;
    (void) argv;

    SDL_Window * window = NULL;
    SDL_Renderer * renderer = NULL;

    SDL_DisplayMode screen;

    // Initialisation de la SDL & Gestion des échecs

    if (SDL_Init(SDL_INIT_VIDEO) != 0) {
        exit_sdl(0, "Error when initializing SDL", window, renderer);
    }

    // Récupération des paramètres de l'écran

    SDL_GetCurrentDisplayMode(0, &screen);

    // Création de la fenêtre

    window = SDL_CreateWindow(
        "Blblblblbl",               // Nom de la fenêtre
        SDL_WINDOWPOS_CENTERED,     // Position x
        SDL_WINDOWPOS_CENTERED,     // Position y
        screen.w * 0.66,            // Largeur de la fenêtre
        screen.h * 0.66,            // Hauteur de la fenêtre
        SDL_WINDOW_OPENGL
    );

    if (window == NULL) {
        exit_sdl(0, "Error when creating the window", window, renderer);
    }

    // Création du renderer

    renderer = SDL_CreateRenderer(window, -1, SDL_RENDERER_ACCELERATED | SDL_RENDERER_PRESENTVSYNC);

    if (renderer == NULL) {
        exit_sdl(0, "Error when creating the renderer", window, renderer);
    }

    // Création des serpents

    srand(time(0));

    int nb_snakes = rand() % MAX_SNAKES;
    int i, j;

    struct snake * snakes = malloc(nb_snakes * sizeof(struct snake));

    if (snakes) {
        for (i = 0; i < nb_snakes; ++i) {
            snakes[i].size = rand() % MAX_PARTS;
            snakes[i].parts = malloc(snakes[i].size * sizeof(struct snake_part));

            double angle = rand() % (int) (2 * PI);

            for (j = 0; j < snakes[i].size; ++j) {

                snakes[i].parts[j].angle = angle;
                snakes[i].parts[j].length = rand() % 50;

                snakes[i].parts[j].color.r = rand() % 255;
                snakes[i].parts[j].color.g = rand() % 255;
                snakes[i].parts[j].color.b = rand() % 255;
            }

            snakes[i].parts[0].x = rand() % (int) (screen.w * 0.66);
            snakes[i].parts[0].y = rand() % (int) (screen.h * 0.66);
        }
    }

    // Déplacement des serpents et affichage

    SDL_bool program_on = SDL_TRUE;
    SDL_bool paused = SDL_FALSE;
    SDL_Event event;

    while (program_on) {
        if (SDL_PollEvent(&event)) {
            switch (event.type) {
                case SDL_QUIT:
                    program_on = SDL_FALSE;
                    break;

                case SDL_KEYDOWN:
                    switch (event.key.keysym.sym) {
                        case SDLK_SPACE:
                            paused = !paused;
                            break;
                        
                        default:
                            break;
                    }
                
                default:
                    break;
            }
        }

        if (!paused) {
            draw_snakes(renderer, snakes, nb_snakes);
            SDL_RenderPresent(renderer); // Affichage de l'image

            SDL_Delay(50);

            SDL_SetRenderDrawColor(renderer, 0, 0, 0, 255);
            SDL_RenderClear(renderer);
        }
    }

    // Fermeture de la SDL

    exit_sdl(1, "Normal ending", window, renderer);
    return EXIT_SUCCESS;
}
